"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.executeWorkflow = void 0;
var _v = require("@kbn/workflows/types/v1");
var _tools = require("@kbn/onechat-common/tools");
var _get_workflow_output = require("./get_workflow_output");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const WORKFLOW_MAX_WAIT = 60_000;
const WORKFLOW_INITIAL_WAIT = 1000;
const WORKFLOW_CHECK_INTERVAL = 2_500;
const executeWorkflow = async ({
  workflowId,
  workflowParams,
  request,
  spaceId,
  workflowApi
}) => {
  const workflow = await workflowApi.getWorkflow(workflowId, spaceId);
  if (!workflow) {
    return [errorResult(`Workflow '${workflowId}' not found.`)];
  }
  if (!workflow.enabled) {
    return [errorResult(`Workflow '${workflowId}' is disabled and cannot be executed.`)];
  }
  if (!workflow.valid) {
    return [errorResult(`Workflow '${workflowId}' has validation errors and cannot be executed.`)];
  }
  if (!workflow.definition) {
    return [errorResult(`Workflow '${workflowId}' has no definition and cannot be executed.`)];
  }
  const executionId = await workflowApi.runWorkflow({
    id: workflow.id,
    name: workflow.name,
    enabled: workflow.enabled,
    definition: workflow.definition,
    yaml: workflow.yaml
  }, spaceId, workflowParams, request);
  const waitStart = Date.now();
  await waitMs(WORKFLOW_INITIAL_WAIT);
  do {
    try {
      const execution = await workflowApi.getWorkflowExecution(executionId, spaceId);
      if (execution) {
        if (execution.status === _v.ExecutionStatus.COMPLETED) {
          const output = (0, _get_workflow_output.getWorkflowOutput)(execution.stepExecutions);
          const data = {
            execution_id: executionId,
            workflow_id: workflowId,
            status: execution.status,
            started_at: execution.startedAt,
            finished_at: execution.finishedAt,
            output
          };
          return [{
            type: _tools.ToolResultType.other,
            data
          }];
        }
        if (execution.status === _v.ExecutionStatus.FAILED) {
          return [{
            type: _tools.ToolResultType.error,
            data: {
              message: `Workflow "${workflow.name}" failed.`,
              metadata: {
                workflow_id: workflowId,
                execution_id: executionId,
                execution_status: execution.status
              }
            }
          }];
        }
      }
    } catch (e) {
      // trap - we just keep waiting until timeout
    }
    await waitMs(WORKFLOW_CHECK_INTERVAL);
  } while (Date.now() - waitStart < WORKFLOW_MAX_WAIT);

  // timeout-ed waiting without completion or failure status
  return [errorResult(`Workflow '${workflowId}' executed but not completed after ${WORKFLOW_MAX_WAIT}ms.`)];
};
exports.executeWorkflow = executeWorkflow;
const errorResult = error => {
  return {
    type: _tools.ToolResultType.error,
    data: {
      message: error
    }
  };
};
const waitMs = async durationMs => {
  await new Promise(resolve => setTimeout(resolve, durationMs));
};