"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addRoundCompleteEvent = void 0;
var _uuid = require("uuid");
var _rxjs = require("rxjs");
var _onechatCommon = require("@kbn/onechat-common");
var _tracing = require("../../../../tracing");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isStepEvent = event => {
  return (0, _onechatCommon.isReasoningEvent)(event) || (0, _onechatCommon.isToolCallEvent)(event);
};
const addRoundCompleteEvent = ({
  userInput,
  startTime,
  endTime,
  modelProvider
}) => {
  return events$ => {
    const shared$ = events$.pipe((0, _rxjs.share)());
    return (0, _rxjs.merge)(shared$, shared$.pipe((0, _rxjs.toArray)(), (0, _rxjs.map)(events => {
      const round = createRoundFromEvents({
        events,
        input: userInput,
        startTime,
        endTime,
        modelProvider
      });
      const event = {
        type: _onechatCommon.ChatEventType.roundComplete,
        data: {
          round
        }
      };
      return event;
    })));
  };
};
exports.addRoundCompleteEvent = addRoundCompleteEvent;
const createRoundFromEvents = ({
  events,
  input,
  startTime,
  endTime = new Date(),
  modelProvider
}) => {
  const toolResults = events.filter(_onechatCommon.isToolResultEvent).map(event => event.data);
  const toolProgressions = events.filter(_onechatCommon.isToolProgressEvent).map(event => event.data);
  const messages = events.filter(_onechatCommon.isMessageCompleteEvent).map(event => event.data);
  const stepEvents = events.filter(isStepEvent);
  const thinkingCompleteEvent = events.find(_onechatCommon.isThinkingCompleteEvent);
  const timeToLastToken = endTime.getTime() - startTime.getTime();
  const timeToFirstToken = thinkingCompleteEvent ? thinkingCompleteEvent.data.time_to_first_token : 0;
  const eventToStep = event => {
    if ((0, _onechatCommon.isToolCallEvent)(event)) {
      var _toolResult$results;
      const toolCall = event.data;
      const toolResult = toolResults.find(result => result.tool_call_id === toolCall.tool_call_id);
      const toolProgress = toolProgressions.filter(progressEvent => progressEvent.tool_call_id === toolCall.tool_call_id).map(progress => ({
        message: progress.message
      }));
      return [{
        type: _onechatCommon.ConversationRoundStepType.toolCall,
        tool_call_id: toolCall.tool_call_id,
        tool_id: toolCall.tool_id,
        progression: toolProgress,
        params: toolCall.params,
        results: (_toolResult$results = toolResult === null || toolResult === void 0 ? void 0 : toolResult.results) !== null && _toolResult$results !== void 0 ? _toolResult$results : []
      }];
    }
    if ((0, _onechatCommon.isReasoningEvent)(event)) {
      if (event.data.transient !== true) {
        return [{
          type: _onechatCommon.ConversationRoundStepType.reasoning,
          reasoning: event.data.reasoning
        }];
      } else {
        return [];
      }
    }
    throw new Error(`Unknown event type: ${event.type}`);
  };
  const round = {
    id: (0, _uuid.v4)(),
    input,
    steps: stepEvents.flatMap(eventToStep),
    trace_id: (0, _tracing.getCurrentTraceId)(),
    started_at: startTime.toISOString(),
    time_to_first_token: timeToFirstToken,
    time_to_last_token: timeToLastToken,
    model_usage: getModelUsage(modelProvider.getUsageStats()),
    response: {
      message: messages[messages.length - 1].message_content
    }
  };
  return round;
};
const getModelUsage = stats => {
  let inputTokens = 0;
  let outputTokens = 0;
  for (const call of stats.calls) {
    var _call$tokens$prompt, _call$tokens, _call$tokens$completi, _call$tokens2;
    inputTokens += (_call$tokens$prompt = (_call$tokens = call.tokens) === null || _call$tokens === void 0 ? void 0 : _call$tokens.prompt) !== null && _call$tokens$prompt !== void 0 ? _call$tokens$prompt : 0;
    outputTokens += (_call$tokens$completi = (_call$tokens2 = call.tokens) === null || _call$tokens2 === void 0 ? void 0 : _call$tokens2.completion) !== null && _call$tokens$completi !== void 0 ? _call$tokens$completi : 0;
  }
  return {
    // we don't support multi-models yet, so we can just pick from the first call
    connector_id: stats.calls.length ? stats.calls[0].connectorId : 'unknown',
    llm_calls: stats.calls.length,
    input_tokens: inputTokens,
    output_tokens: outputTokens
  };
};