"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.extractToolReturn = exports.convertGraphEvents = void 0;
var _uuid = require("uuid");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _langchain = require("@kbn/onechat-genai-utils/langchain");
var _onechatServer = require("@kbn/onechat-server");
var _constants = require("./constants");
var _actions = require("./actions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const convertGraphEvents = ({
  graphName,
  toolIdMapping,
  logger,
  startTime
}) => {
  return streamEvents$ => {
    const toolCallIdToIdMap = new Map();
    const messageId = (0, _uuid.v4)();
    let isThinkingComplete = false;
    return streamEvents$.pipe((0, _rxjs.mergeMap)(event => {
      if (!(0, _langchain.matchGraphName)(event, graphName)) {
        return _rxjs.EMPTY;
      }

      // stream answering text chunks for the UI
      if ((0, _langchain.matchEvent)(event, 'on_chat_model_stream') && (0, _langchain.hasTag)(event, _constants.tags.answerAgent)) {
        const chunk = event.data.chunk;
        const textContent = (0, _langchain.extractTextContent)(chunk);
        if (textContent) {
          const events = [];
          if (!isThinkingComplete) {
            // Emit thinking complete event when first chunk arrives
            events.push((0, _langchain.createThinkingCompleteEvent)(Date.now() - startTime.getTime()));
            isThinkingComplete = true;
          }
          events.push((0, _langchain.createTextChunkEvent)(textContent, {
            messageId
          }));
          return (0, _rxjs.of)(...events);
        }
      }

      // emit tool calls for research agent steps
      if ((0, _langchain.matchEvent)(event, 'on_chain_end') && (0, _langchain.matchName)(event, _constants.steps.researchAgent)) {
        const events = [];
        const addedActions = event.data.output.mainActions;
        const nextAction = addedActions[addedActions.length - 1];
        if ((0, _actions.isToolCallAction)(nextAction)) {
          const {
            tool_calls: toolCalls,
            message: messageText
          } = nextAction;
          if (toolCalls.length > 0) {
            let hasReasoningEvent = false;
            for (const toolCall of toolCalls) {
              const toolId = (0, _langchain.toolIdentifierFromToolCall)(toolCall, toolIdMapping);
              const {
                toolCallId,
                args
              } = toolCall;
              const {
                _reasoning,
                ...toolCallArgs
              } = args;
              if (_reasoning) {
                events.push((0, _langchain.createReasoningEvent)(_reasoning));
                hasReasoningEvent = true;
              }
              toolCallIdToIdMap.set(toolCall.toolCallId, toolId);
              const isBrowserTool = toolId.startsWith(_constants.BROWSER_TOOL_PREFIX);
              if (isBrowserTool) {
                events.push((0, _langchain.createBrowserToolCallEvent)({
                  toolId: toolId.replace(_constants.BROWSER_TOOL_PREFIX, ''),
                  toolCallId,
                  params: toolCallArgs
                }));
              } else {
                events.push((0, _langchain.createToolCallEvent)({
                  toolId,
                  toolCallId,
                  params: toolCallArgs
                }));
              }
            }
            if (messageText && !hasReasoningEvent) {
              events.push((0, _langchain.createReasoningEvent)(messageText));
            }
          }
        }
        return (0, _rxjs.of)(...events);
      }

      // emit messages for answering step
      if ((0, _langchain.matchEvent)(event, 'on_chain_end') && (0, _langchain.matchName)(event, _constants.steps.answerAgent)) {
        const events = [];

        // process last emitted message
        const answerActions = event.data.output.answerActions;
        const lastAction = answerActions[answerActions.length - 1];
        if ((0, _actions.isAnswerAction)(lastAction)) {
          const messageEvent = (0, _langchain.createMessageEvent)(lastAction.message, {
            messageId
          });
          events.push(messageEvent);
        }
        return (0, _rxjs.of)(...events);
      }

      // emit tool result events
      if ((0, _langchain.matchEvent)(event, 'on_chain_end') && (0, _langchain.matchName)(event, _constants.steps.executeTool)) {
        const addedActions = event.data.output.mainActions;
        const nextAction = addedActions[addedActions.length - 1];
        if ((0, _actions.isExecuteToolAction)(nextAction)) {
          const toolResultEvents = [];
          for (const toolResult of nextAction.tool_results) {
            const toolId = toolCallIdToIdMap.get(toolResult.toolCallId);
            const toolReturn = extractToolReturn(toolResult);
            toolResultEvents.push((0, _langchain.createToolResultEvent)({
              toolCallId: toolResult.toolCallId,
              toolId: toolId !== null && toolId !== void 0 ? toolId : 'unknown',
              results: toolReturn.results
            }));
          }
          return (0, _rxjs.of)(...toolResultEvents);
        }
      }
      return _rxjs.EMPTY;
    }));
  };
};
exports.convertGraphEvents = convertGraphEvents;
const extractToolReturn = message => {
  if (message.artifact) {
    if (!(0, _lodash.isArray)(message.artifact.results)) {
      throw new Error(`Artifact is not a structured tool artifact. Received artifact=${JSON.stringify(message.artifact)}`);
    }
    return message.artifact;
  } else {
    // langchain tool validation error (such as schema errors) are out of our control and don't emit artifacts...
    if (message.content.startsWith('Error:')) {
      return {
        results: [(0, _onechatServer.createErrorResult)(message.content)]
      };
    } else {
      throw new Error(`No artifact attached to tool message: ${JSON.stringify(message)}`);
    }
  }
};
exports.extractToolReturn = extractToolReturn;