"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerDocumentationFunction = registerDocumentationFunction;
var _productDocCommon = require("@kbn/product-doc-common");
var _server = require("@kbn/observability-ai-assistant-plugin/server");
var _inferenceCommon = require("@kbn/inference-common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function registerDocumentationFunction({
  functions,
  resources,
  pluginsStart: {
    llmTasks
  }
}) {
  var _await$getInferenceId, _await$llmTasks$retri;
  const esClient = (await resources.context.core).elasticsearch.client;
  const inferenceId = (_await$getInferenceId = await (0, _server.getInferenceIdFromWriteIndex)(esClient, resources.logger)) !== null && _await$getInferenceId !== void 0 ? _await$getInferenceId : _inferenceCommon.defaultInferenceEndpoints.ELSER;
  const isProductDocAvailable = (_await$llmTasks$retri = await llmTasks.retrieveDocumentationAvailable({
    inferenceId
  })) !== null && _await$llmTasks$retri !== void 0 ? _await$llmTasks$retri : false;
  functions.registerFunction({
    name: _server.RETRIEVE_ELASTIC_DOC_FUNCTION_NAME,
    isInternal: !isProductDocAvailable,
    description: `Use this function to retrieve documentation about Elastic products.
      You can retrieve documentation about the Elastic stack, such as Kibana and Elasticsearch,
      or for Elastic solutions, such as Elastic Security, Elastic Observability or Elastic Enterprise Search
      `,
    parameters: {
      type: 'object',
      properties: {
        query: {
          description: `The query to use to retrieve documentation
            Always write the query in English, as the documentation is available only in English.
            Examples:
            - "How to enable TLS for Elasticsearch?"
            - "What is Kibana Lens?"`,
          type: 'string'
        },
        product: {
          description: `If specified, will filter the products to retrieve documentation for
            Possible options are:
            - "kibana": Kibana product
            - "elasticsearch": Elasticsearch product
            - "observability": Elastic Observability solution
            - "security": Elastic Security solution
            If not specified, will search against all products
            `,
          type: 'string',
          enum: Object.values(_productDocCommon.DocumentationProduct)
        }
      },
      required: ['query']
    }
  }, async ({
    arguments: {
      query,
      product
    },
    connectorId,
    simulateFunctionCalling
  }) => {
    const response = await llmTasks.retrieveDocumentation({
      searchTerm: query,
      products: product ? [product] : undefined,
      max: 3,
      connectorId,
      request: resources.request,
      functionCalling: simulateFunctionCalling ? 'simulated' : 'auto',
      inferenceId
    });
    return {
      content: {
        documents: response.documents
      }
    };
  });
}