"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildDatasourceStates = exports.buildDatasetState = exports.addLayerColumn = void 0;
exports.buildReferences = buildReferences;
exports.createDataViewReference = createDataViewReference;
exports.getAdhocDataviews = exports.generateLayer = exports.generateApiLayer = exports.filtersToLensState = exports.filtersToApiFormat = exports.filtersAndQueryToLensState = exports.filtersAndQueryToApiFormat = void 0;
exports.getDatasetIndex = getDatasetIndex;
exports.getDefaultReferences = void 0;
exports.isSingleLayer = isSingleLayer;
exports.isTextBasedLayer = isTextBasedLayer;
exports.nonNullable = nonNullable;
exports.queryToLensState = exports.queryToApiFormat = exports.operationFromColumn = void 0;
var _uuid = require("uuid");
var _esqlUtils = require("@kbn/esql-utils");
var _esQuery = require("@kbn/es-query");
var _buckets = require("./columns/buckets");
var _metric = require("./columns/metric");
var _utils = require("./columns/utils");
var _constants = require("./constants");
var _constants2 = require("../schema/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function createDataViewReference(index, layerId) {
  return {
    type: _constants.INDEX_PATTERN_ID,
    id: index,
    name: `${_constants.LENS_LAYER_SUFFIX}${layerId}`
  };
}
const getDefaultReferences = (index, dataLayerId) => {
  return [createDataViewReference(index, dataLayerId)];
};

// Need to dance a bit to satisfy TypeScript
exports.getDefaultReferences = getDefaultReferences;
function convertToTypedLayerColumns(layer) {
  // @TODO move it to satisfies
  return layer;
}

/**
 * given Lens State layer and column id, returns the corresponding Lens API operation
 * @param columnId
 * @param layer
 * @returns
 */
const operationFromColumn = (columnId, layer) => {
  const typedLayer = convertToTypedLayerColumns(layer);
  const column = typedLayer.columns[columnId];
  if (!column) {
    return undefined;
  }
  // map columns to array of { column, id }
  const columnMap = Object.entries(layer.columns).map(([id, c]) => ({
    column: c,
    id
  }));
  if ((0, _utils.isLensStateBucketColumnType)(column)) {
    return (0, _buckets.fromBucketLensStateToAPI)(column, columnMap);
  }
  return (0, _metric.getMetricApiColumnFromLensState)(column, typedLayer.columns);
};
exports.operationFromColumn = operationFromColumn;
function isTextBasedLayer(layer) {
  return 'index' in layer && 'query' in layer;
}
function getAdHocDataViewSpec(dataView) {
  return {
    id: (0, _uuid.v4)({}),
    title: dataView.index,
    name: dataView.index,
    timeFieldName: dataView.timeFieldName,
    sourceFilters: [],
    fieldFormats: {},
    runtimeFieldMap: {},
    fieldAttrs: {},
    allowNoIndex: false,
    allowHidden: false
  };
}
const getAdhocDataviews = dataviews => {
  // filter out ad hoc dataViews only
  const adHocDataViewsFiltered = Object.entries(dataviews).filter(([_layerId, dataViewEntry]) => dataViewEntry.type === 'adHocDataView');
  const internalReferencesMap = new Map();

  // dedupe and map multiple layer references to the same ad hoc dataview
  for (const [layerId, dataViewEntry] of adHocDataViewsFiltered) {
    if (!internalReferencesMap.has(dataViewEntry)) {
      internalReferencesMap.set(dataViewEntry, {
        layerIds: [],
        id: (0, _uuid.v4)({})
      });
    }
    const internalRef = internalReferencesMap.get(dataViewEntry);
    internalRef.layerIds.push(layerId);
  }
  const adHocDataViews = {};
  const internalReferences = [];
  for (const [baseSpec, {
    layerIds,
    id
  }] of internalReferencesMap.entries()) {
    adHocDataViews[id] = getAdHocDataViewSpec(baseSpec);
    for (const layerId of layerIds) {
      internalReferences.push(createDataViewReference(id, layerId));
    }
  }
  return {
    adHocDataViews,
    internalReferences
  };
};

/**
 * Builds dataset state from the layer configuration
 *
 * @param layer Lens State Layer
 * @returns Lens API Dataset configuration
 */
exports.getAdhocDataviews = getAdhocDataviews;
const buildDatasetState = (layer, adHocDataViews, references, adhocReferences = [], layerId) => {
  if (isTextBasedLayer(layer)) {
    var _layer$query$esql, _layer$query;
    return {
      type: 'esql',
      query: (_layer$query$esql = (_layer$query = layer.query) === null || _layer$query === void 0 ? void 0 : _layer$query.esql) !== null && _layer$query$esql !== void 0 ? _layer$query$esql : ''
    };
  }
  const adhocReference = (adhocReferences !== null && adhocReferences !== void 0 ? adhocReferences : []).find(ref => ref.name === `${_constants.LENS_LAYER_SUFFIX}${layerId}`);
  if (adhocReference && adHocDataViews !== null && adHocDataViews !== void 0 && adHocDataViews[adhocReference.id]) {
    var _adHocDataViews$adhoc;
    return {
      type: 'index',
      index: adHocDataViews[adhocReference.id].title,
      time_field: (_adHocDataViews$adhoc = adHocDataViews[adhocReference.id].timeFieldName) !== null && _adHocDataViews$adhoc !== void 0 ? _adHocDataViews$adhoc : _constants.LENS_DEFAULT_TIME_FIELD
    };
  }
  const reference = (references !== null && references !== void 0 ? references : []).find(ref => ref.name === `${_constants.LENS_LAYER_SUFFIX}${layerId}`);
  if (reference) {
    return {
      type: 'dataView',
      id: reference.id
    };
  }
  return {
    type: 'dataView',
    id: layer.indexPatternId
  };
};

// builds Lens State references from list of dataviews
exports.buildDatasetState = buildDatasetState;
function buildReferences(dataviews) {
  const references = [];
  for (const layerid in dataviews) {
    if (dataviews[layerid]) {
      references.push(getDefaultReferences(dataviews[layerid], layerid));
    }
  }
  return references.flat(2);
}
function isSingleLayer(layer) {
  return layer && typeof layer === 'object' && ('columnOrder' in layer || 'columns' in layer);
}

/**
 * Gets DataView from the dataset configuration
 *
 * @param dataset
 * @param dataViewsAPI
 * @returns
 */
function getDatasetIndex(dataset) {
  const timeFieldName = _constants.LENS_DEFAULT_TIME_FIELD;
  switch (dataset.type) {
    case 'index':
      return {
        index: dataset.index,
        timeFieldName
      };
    case 'esql':
      return {
        index: (0, _esqlUtils.getIndexPatternFromESQLQuery)(dataset.query),
        timeFieldName
      };
    case 'dataView':
      return {
        index: dataset.id,
        timeFieldName
      };
    case 'table':
      return;
    default:
      throw Error('dataset type not supported');
  }
}

// internal function used to build datasource states layer
function buildDatasourceStatesLayer(layer, i, dataset, index, buildDataLayers, getValueColumns) {
  function buildValueLayer(config, ds) {
    const table = ds.table;
    const newLayer = {
      table,
      columns: getValueColumns(config, i),
      allColumns: table.columns.map(column => ({
        fieldName: column.name,
        columnId: column.id,
        meta: column.meta
      })),
      index: '',
      query: undefined
    };
    return newLayer;
  }
  function buildESQLLayer(config, ds) {
    const columns = getValueColumns(config, i);
    return {
      index: index.index,
      query: {
        esql: ds.query
      },
      timeField: _constants.LENS_DEFAULT_TIME_FIELD,
      columns
    };
  }
  if (dataset.type === 'esql') {
    return ['textBased', buildESQLLayer(layer, dataset)];
  }
  if (dataset.type === 'table') {
    return ['textBased', buildValueLayer(layer, dataset)];
  }
  return ['formBased', buildDataLayers(layer, i, index)];
}

/**
 * Builds lens config datasource states from LensApiState
 *
 * @param config lens api state
 * @param dataviews list to which dataviews are added
 * @param buildFormulaLayers function used when dataset type is index or dataView
 * @param getValueColumns function used when dataset type is table or esql
 * @param dataViewsAPI dataViews service
 * @returns lens datasource states
 *
 */
const buildDatasourceStates = (config, buildFormulaLayers, getValueColumns) => {
  let layers = {};
  const mainDataset = config.dataset;
  const usedDataviews = {};
  // a few charts types support multiple layers
  const configLayers = 'layers' in config ? config.layers : [config];
  for (let i = 0; i < configLayers.length; i++) {
    const layer = configLayers[i];
    const layerId = `layer_${i}`;
    const dataset = layer.dataset || mainDataset;
    if (!dataset) {
      throw Error('dataset must be defined');
    }
    const index = getDatasetIndex(dataset);
    const [type, layerConfig] = buildDatasourceStatesLayer(layer, i, dataset, index, buildFormulaLayers, getValueColumns);
    if (layerConfig) {
      var _layers$type;
      layers = {
        ...layers,
        [type]: {
          layers: isSingleLayer(layerConfig) ? {
            ...((_layers$type = layers[type]) === null || _layers$type === void 0 ? void 0 : _layers$type.layers),
            [layerId]: layerConfig
          } :
          // metric chart can return 2 layers (one for the metric and one for the trendline)
          {
            ...layerConfig
          }
        }
      };
    }

    // keep record of all dataviews used by layers
    if (index) {
      var _layers$type$layers, _layers$type2;
      Object.keys((_layers$type$layers = (_layers$type2 = layers[type]) === null || _layers$type2 === void 0 ? void 0 : _layers$type2.layers) !== null && _layers$type$layers !== void 0 ? _layers$type$layers : []).forEach(id => {
        usedDataviews[id] = dataset.type === 'dataView' ? {
          type: 'dataView',
          id: dataset.id
        } : {
          type: 'adHocDataView',
          ...index
        };
      });
    }
  }
  return {
    layers,
    usedDataviews
  };
};

// adds new column to existing layer
exports.buildDatasourceStates = buildDatasourceStates;
const addLayerColumn = (layer, columnName, config, first = false, postfix = '') => {
  const [column, referenceColumn] = Array.isArray(config) ? config : [config];
  const name = columnName + postfix;
  const referenceColumnId = `${name}_reference`;
  layer.columns = {
    ...layer.columns,
    [name]: column,
    ...(referenceColumn ? {
      [referenceColumnId]: referenceColumn
    } : {})
  };
  if (first) {
    layer.columnOrder.unshift(name);
    if (referenceColumn) {
      layer.columnOrder.unshift(referenceColumnId);
    }
  } else {
    layer.columnOrder.push(name);
    if (referenceColumn) {
      layer.columnOrder.push(referenceColumnId);
    }
  }
};

/**
 * Generates the base layer
 *
 * @param id
 * @param options
 * @returns
 */
exports.addLayerColumn = addLayerColumn;
const generateLayer = (id, options) => {
  return {
    [id]: {
      sampling: options.sampling,
      ignoreGlobalFilters: options.ignore_global_filters,
      columns: {},
      columnOrder: []
    }
  };
};
exports.generateLayer = generateLayer;
const generateApiLayer = options => {
  var _options$sampling, _options$ignoreGlobal;
  if (!('columnOrder' in options)) {
    return {
      sampling: _constants2.LENS_SAMPLING_DEFAULT_VALUE,
      ignore_global_filters: _constants2.LENS_IGNORE_GLOBAL_FILTERS_DEFAULT_VALUE
    };
  }
  // mind this is already filled by schema validate
  return {
    sampling: (_options$sampling = options.sampling) !== null && _options$sampling !== void 0 ? _options$sampling : _constants2.LENS_SAMPLING_DEFAULT_VALUE,
    ignore_global_filters: (_options$ignoreGlobal = options.ignoreGlobalFilters) !== null && _options$ignoreGlobal !== void 0 ? _options$ignoreGlobal : _constants2.LENS_IGNORE_GLOBAL_FILTERS_DEFAULT_VALUE
  };
};
exports.generateApiLayer = generateApiLayer;
const filtersToApiFormat = filters => {
  return filters.map(filter => {
    var _filter$query, _filter$query$query, _filter$query2;
    return {
      language: (_filter$query = filter.query) === null || _filter$query === void 0 ? void 0 : _filter$query.language,
      query: (_filter$query$query = (_filter$query2 = filter.query) === null || _filter$query2 === void 0 ? void 0 : _filter$query2.query) !== null && _filter$query$query !== void 0 ? _filter$query$query : filter.query,
      meta: {}
    };
  });
};
exports.filtersToApiFormat = filtersToApiFormat;
const queryToApiFormat = query => {
  if (typeof query.query !== 'string') {
    return;
  }
  return {
    query: query.query,
    language: query.language
  };
};
exports.queryToApiFormat = queryToApiFormat;
const filtersToLensState = filters => {
  return filters.map(filter => ({
    query: {
      query: filter.query,
      language: filter === null || filter === void 0 ? void 0 : filter.language
    },
    meta: {}
  }));
};
exports.filtersToLensState = filtersToLensState;
const queryToLensState = query => {
  return query;
};
exports.queryToLensState = queryToLensState;
const filtersAndQueryToApiFormat = state => {
  var _state$state$filters;
  return {
    ...((_state$state$filters = state.state.filters) !== null && _state$state$filters !== void 0 && _state$state$filters.length ? {
      filters: filtersToApiFormat(state.state.filters)
    } : {}),
    ...(state.state.query && !(0, _esQuery.isOfAggregateQueryType)(state.state.query) ? {
      query: queryToApiFormat(state.state.query)
    } : {})
  };
};
exports.filtersAndQueryToApiFormat = filtersAndQueryToApiFormat;
const filtersAndQueryToLensState = state => {
  const query = state.dataset.type === 'esql' ? {
    esql: state.dataset.query
  } : 'query' in state && state.query ? queryToLensState(state.query) : undefined;
  return {
    ...(state.filters ? {
      filters: filtersToLensState(state.filters)
    } : {}),
    ...(query ? {
      query
    } : {})
  };
};
exports.filtersAndQueryToLensState = filtersAndQueryToLensState;
function nonNullable(v) {
  return v != null;
}