"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.executeAsReasoningAgent = executeAsReasoningAgent;
var _inferenceCommon = require("@kbn/inference-common");
var _inferenceTracing = require("@kbn/inference-tracing");
var _api = require("@opentelemetry/api");
var _lodash = require("lodash");
var _zod = require("@kbn/zod");
var _create_complete_tool_call = require("./create_complete_tool_call");
var _create_reason_tool_call = require("./create_reason_tool_call");
var _format_for_power = require("./format_for_power");
var _markers = require("./markers");
var _planning_tools = require("./planning_tools");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Executes a prompt in a loop in a way that the LLM will use the specified tools
 * to gather context, and then produce a final output, which may or may not include
 * a final tool call.
 *
 * The rules are as follows:
 * - when `reason()` is called, the LLM SHOULD reason about the task or the tool call
 * results
 * - when `reason()` is called, the LLM CAN call another tool
 * - when `complete()` is called, and `finalToolChoice` is NOT specified, the LLM
 * MUST produce a summarization text
 * - when `complete()` is called, and `finalToolChoice` is specified, the LLM MUST
 * call a tool to complete the task, and the LLM SHOULD produce a summarization text
 * - when `finalToolChoice` is specified, and the LLM calls this tool, the task MUST
 * be completed by the orchestrator
 * - when the available number of steps have been exhausted, the LLM MUST produce
 * its final output
 * - if the LLM fails to produce its final output (e.g. by calling an unavailable tool),
 * the orchestrator MUST complete the task
 * - if `finalToolChoice` is specified and is not included as part of the definitive output,
 * the orchestrator MUST fail the task
 */
async function executeAsReasoningAgent(options) {
  const {
    inferenceClient,
    maxSteps = 10,
    power = 'medium',
    toolCallbacks
  } = options;
  async function callTools(toolCalls) {
    return await Promise.all(toolCalls.map(async toolCall => {
      if ((0, _planning_tools.isPlanningToolName)(toolCall.function.name)) {
        throw new Error(`Unexpected planning tool call ${toolCall.function.name}`);
      }
      const callback = toolCallbacks[toolCall.function.name];
      const response = await (0, _inferenceTracing.withExecuteToolSpan)(toolCall.function.name, {
        tool: {
          input: 'arguments' in toolCall.function ? toolCall.function.arguments : undefined,
          toolCallId: toolCall.toolCallId
        }
      }, () => callback(toolCall)).catch(error => {
        var _trace$getActiveSpan;
        (_trace$getActiveSpan = _api.trace.getActiveSpan()) === null || _trace$getActiveSpan === void 0 ? void 0 : _trace$getActiveSpan.recordException(error);
        return {
          response: {
            error,
            data: undefined
          }
        };
      });
      return {
        response: response.response,
        data: response.data,
        name: toolCall.function.name,
        toolCallId: toolCall.toolCallId,
        role: _inferenceCommon.MessageRole.Tool
      };
    }));
  }
  async function innerCallPromptUntil({
    messages: givenMessages,
    stepsLeft,
    temperature
  }) {
    var _givenMessages$findLa, _lastAssistantMessage;
    const lastAssistantMessage = givenMessages.findLast(msg => msg.role === _inferenceCommon.MessageRole.Assistant);
    const lastSystemToolCallName = (_givenMessages$findLa = givenMessages.findLast(message => message.role === _inferenceCommon.MessageRole.Tool && (0, _planning_tools.isPlanningToolName)(message.name))) === null || _givenMessages$findLa === void 0 ? void 0 : _givenMessages$findLa.name;
    const shouldComplete = stepsLeft <= 0 || lastSystemToolCallName === 'complete';

    // reason when:
    // - not completing
    // - AND power is medium or high
    // - AND last assistant message contains a `reason` tool call
    const shouldReason = !shouldComplete && (power === 'medium' || power === 'high') && (lastAssistantMessage === null || lastAssistantMessage === void 0 ? void 0 : (_lastAssistantMessage = lastAssistantMessage.toolCalls) === null || _lastAssistantMessage === void 0 ? void 0 : _lastAssistantMessage.some(toolCall => (0, _planning_tools.isPlanningToolName)(toolCall.function.name)));
    const prevMessages = givenMessages.concat();

    // these are hints
    if (shouldComplete && lastSystemToolCallName !== 'complete') {
      prevMessages.push(...(0, _create_complete_tool_call.createCompleteToolCall)());
    } else if (shouldReason && lastSystemToolCallName !== 'reason') {
      prevMessages.push(...(0, _create_reason_tool_call.createReasonToolCall)());
    }
    const nextPrompt = {
      ...options.prompt,
      versions: options.prompt.versions.map(version => {
        const {
          tools: promptTools,
          ...rest
        } = version;
        if (power === 'low') {
          return {
            ...rest,
            tools: promptTools
          };
        }
        return (0, _format_for_power.formatToolOptions)({
          ...rest,
          tools: {
            ...promptTools,
            ..._planning_tools.PLANNING_TOOLS
          }
        }, power);
      })
    };
    const forceComplete = shouldComplete;
    const forceReason = power === 'high' && shouldReason;
    const promptOptions = {
      ...(0, _lodash.omit)(options, 'finalToolChoice'),
      prompt: nextPrompt
    };
    const toolChoice = forceComplete ? options.finalToolChoice || _inferenceCommon.ToolChoiceType.none : forceReason ? _inferenceCommon.ToolChoiceType.none : _inferenceCommon.ToolChoiceType.auto;
    const response = await inferenceClient.prompt({
      ...promptOptions,
      prompt: {
        ...promptOptions.prompt,
        input: _zod.z.intersection(promptOptions.prompt.input, _zod.z.object({
          power: _zod.z.object({
            low: _zod.z.boolean(),
            medium: _zod.z.boolean(),
            high: _zod.z.boolean()
          })
        }))
      },
      input: {
        ...promptOptions.input,
        power: {
          low: power === 'low',
          medium: power === 'medium',
          high: power === 'high'
        }
      },
      stream: false,
      temperature,
      toolChoice,
      prevMessages: (0, _format_for_power.formatMessages)({
        messages: prevMessages,
        power,
        stepsLeft
      }),
      stopSequences: [_markers.END_INTERNAL_REASONING_MARKER]
    });
    let content = response.content;

    /**
     * If the LLM hasn't used these markers, we assume it wants to complete its
     * output.
     */

    let completeNextTurn = content && !content.includes(_markers.BEGIN_INTERNAL_REASONING_MARKER) && !content.includes(_markers.END_INTERNAL_REASONING_MARKER) && !response.toolCalls.length;

    /**
     * Remove content after <<<END_INTERNAL>>>. This means that the LLM has combined final output
     * with internal reasoning, and it usually leads the LLM into a loop where it repeats itself.
     */

    const [internalContent, ...externalContentParts] = content.split(_markers.END_INTERNAL_REASONING_MARKER);
    const externalContent = externalContentParts.join(_markers.END_INTERNAL_REASONING_MARKER).trim();

    // use some kind of buffer to allow small artifacts around the markers, like markdown.
    if (externalContent.length && externalContent.length > 25) {
      content = internalContent + _markers.END_INTERNAL_REASONING_MARKER;
      completeNextTurn = true;
    }
    const assistantMessage = {
      role: _inferenceCommon.MessageRole.Assistant,
      content,
      toolCalls: response.toolCalls
    };
    const [systemToolCalls, nonSystemToolCalls] = (0, _lodash.partition)(response.toolCalls, toolCall => (0, _planning_tools.isPlanningToolName)(toolCall.function.name));
    if (systemToolCalls.length && response.toolCalls.length > 1) {
      throw new Error(`When using system tools, only a single tool call is allowed`);
    }
    const finalToolCallName = options.finalToolChoice && typeof options.finalToolChoice === 'object' ? options.finalToolChoice.function : undefined;
    const hasCalledFinalTool = response.toolCalls.some(toolCall => toolCall.function.name === finalToolCallName);
    if (shouldComplete || hasCalledFinalTool) {
      // We don't want to send these results back to the LLM, if we are already
      // completing
      return {
        content: response.content,
        tokens: response.tokens,
        toolCalls: response.toolCalls.filter(toolCall => toolCall.function.name === finalToolCallName),
        input: (0, _planning_tools.removeSystemToolCalls)(prevMessages)
      };
    }
    const toolMessagesForNonSystemToolCalls = nonSystemToolCalls.length ? (await callTools(nonSystemToolCalls)).map(toolMessage => {
      return {
        ...toolMessage,
        response: {
          ...(typeof toolMessage.response === 'string' ? {
            content: toolMessage.response
          } : toolMessage.response),
          stepsLeft
        }
      };
    }) : [];
    const systemToolMessages = systemToolCalls.map(systemToolCall => {
      if (systemToolCall.function.name === 'reason') {
        return (0, _create_reason_tool_call.createReasonToolCallResponse)(systemToolCall.toolCallId);
      }
      return (0, _create_complete_tool_call.createCompleteToolCallResponse)(systemToolCall.toolCallId);
    });
    const allToolMessages = [...toolMessagesForNonSystemToolCalls, ...systemToolMessages];
    if (completeNextTurn) {
      return innerCallPromptUntil({
        messages: prevMessages.concat(assistantMessage, ...allToolMessages),
        stepsLeft: 0
      });
    }
    return innerCallPromptUntil({
      messages: prevMessages.concat(assistantMessage, ...allToolMessages, ...(nonSystemToolCalls.length ? (0, _create_reason_tool_call.createReasonToolCall)() : [])),
      stepsLeft: stepsLeft - 1
    });
  }
  return await (0, _inferenceTracing.withActiveInferenceSpan)('reason', () => innerCallPromptUntil({
    // nudge the LLM to go into reasoning mode
    messages: (0, _create_reason_tool_call.createReasonToolCall)(),
    stepsLeft: maxSteps
  }));
}