"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.toAsyncIterator = toAsyncIterator;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Convert an Observable into an async iterator.
 * (don't ask, langchain is using async iterators for stream mode...)
 */
function toAsyncIterator(observable) {
  let resolve = null;
  let reject = null;
  const queue = [];
  let done = false;
  let error = null;
  const subscription = observable.subscribe({
    next(value) {
      if (resolve) {
        resolve({
          value,
          done: false
        });
        resolve = null;
      } else {
        queue.push({
          value,
          done: false
        });
      }
    },
    error(err) {
      done = true;
      error = err;
      // Clear any queued values - we fail fast
      queue.length = 0;
      if (reject) {
        reject(err);
        reject = null;
        resolve = null;
      }
    },
    complete() {
      done = true;
      if (resolve) {
        resolve({
          value: undefined,
          done: true
        });
        resolve = null;
      }
    }
  });
  return {
    [Symbol.asyncIterator]() {
      return this;
    },
    next() {
      // Check for error first - fail fast
      if (error !== null) {
        return Promise.reject(error);
      }
      if (queue.length > 0) {
        return Promise.resolve(queue.shift());
      }
      if (done) {
        return Promise.resolve({
          value: undefined,
          done: true
        });
      }
      return new Promise((res, rej) => {
        resolve = res;
        reject = rej;
      });
    },
    return() {
      subscription.unsubscribe();
      return Promise.resolve({
        value: undefined,
        done: true
      });
    },
    throw(err) {
      subscription.unsubscribe();
      return Promise.reject(err);
    }
  };
}