"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports._stateMachineInstallPackage = _stateMachineInstallPackage;
var _server = require("@kbn/core/server");
var _errors = require("../../../../errors");
var _types = require("../../../../../common/types");
var _ = require("../../..");
var _steps = require("./steps");
var _state_machine = require("./state_machine");
var _step_create_alerting_rules = require("./steps/step_create_alerting_rules");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * This data structure defines the sequence of the states and the transitions
 */
const regularStatesDefinition = {
  create_restart_installation: {
    nextState: _types.INSTALL_STATES.INSTALL_PRECHECK,
    onTransition: _steps.stepCreateRestartInstallation,
    onPostTransition: _steps.updateLatestExecutedState
  },
  install_precheck: {
    onTransition: _steps.stepInstallPrecheck,
    nextState: _types.INSTALL_STATES.INSTALL_KIBANA_ASSETS,
    onPostTransition: _steps.updateLatestExecutedState
  },
  install_kibana_assets: {
    onPreTransition: _steps.cleanUpKibanaAssetsStep,
    onTransition: _steps.stepInstallKibanaAssets,
    nextState: _types.INSTALL_STATES.INSTALL_ILM_POLICIES,
    onPostTransition: _steps.updateLatestExecutedState
  },
  install_ilm_policies: {
    onPreTransition: _steps.cleanupILMPoliciesStep,
    onTransition: _steps.stepInstallILMPolicies,
    nextState: _types.INSTALL_STATES.INSTALL_ML_MODEL,
    onPostTransition: _steps.updateLatestExecutedState
  },
  install_ml_model: {
    onPreTransition: _steps.cleanUpMlModelStep,
    onTransition: _steps.stepInstallMlModel,
    nextState: _types.INSTALL_STATES.INSTALL_INDEX_TEMPLATE_PIPELINES,
    onPostTransition: _steps.updateLatestExecutedState
  },
  install_index_template_pipelines: {
    onPreTransition: _steps.cleanupIndexTemplatePipelinesStep,
    onTransition: _steps.stepInstallIndexTemplatePipelines,
    nextState: _types.INSTALL_STATES.REMOVE_LEGACY_TEMPLATES,
    onPostTransition: _steps.updateLatestExecutedState
  },
  remove_legacy_templates: {
    onTransition: _steps.stepRemoveLegacyTemplates,
    nextState: _types.INSTALL_STATES.UPDATE_CURRENT_WRITE_INDICES,
    onPostTransition: _steps.updateLatestExecutedState
  },
  update_current_write_indices: {
    onTransition: _steps.stepUpdateCurrentWriteIndices,
    nextState: _types.INSTALL_STATES.INSTALL_TRANSFORMS,
    onPostTransition: _steps.updateLatestExecutedState
  },
  install_transforms: {
    onPreTransition: _steps.cleanupTransformsStep,
    onTransition: _steps.stepInstallTransforms,
    nextState: _types.INSTALL_STATES.DELETE_PREVIOUS_PIPELINES,
    onPostTransition: _steps.updateLatestExecutedState
  },
  delete_previous_pipelines: {
    onTransition: _steps.stepDeletePreviousPipelines,
    nextState: _types.INSTALL_STATES.SAVE_ARCHIVE_ENTRIES,
    onPostTransition: _steps.updateLatestExecutedState
  },
  save_archive_entries_from_assets_map: {
    onPreTransition: _steps.cleanupArchiveEntriesStep,
    onTransition: _steps.stepSaveArchiveEntries,
    nextState: _types.INSTALL_STATES.SAVE_KNOWLEDGE_BASE,
    onPostTransition: _steps.updateLatestExecutedState
  },
  save_knowledge_base: {
    onPreTransition: _steps.cleanupKnowledgeBaseStep,
    onTransition: _steps.stepSaveKnowledgeBase,
    nextState: _types.INSTALL_STATES.RESOLVE_KIBANA_PROMISE,
    onPostTransition: _steps.updateLatestExecutedState,
    isAsync: true // Knowledge base indexing runs in background
  },
  resolve_kibana_promise: {
    onTransition: _steps.stepResolveKibanaPromise,
    nextState: _types.INSTALL_STATES.CREATE_ALERTING_RULES,
    onPostTransition: _steps.updateLatestExecutedState
  },
  create_alerting_rules: {
    onTransition: _step_create_alerting_rules.stepCreateAlertingRules,
    nextState: _types.INSTALL_STATES.UPDATE_SO,
    onPostTransition: _steps.updateLatestExecutedState
  },
  update_so: {
    onTransition: _steps.stepSaveSystemObject,
    nextState: 'end',
    onPostTransition: _steps.updateLatestExecutedState
  }
};
const streamingStatesDefinition = {
  create_restart_installation: {
    nextState: _types.INSTALL_STATES.INSTALL_KIBANA_ASSETS,
    onTransition: _steps.stepCreateRestartInstallation,
    onPostTransition: _steps.updateLatestExecutedState
  },
  install_kibana_assets: {
    onTransition: _steps.stepInstallKibanaAssetsWithStreaming,
    nextState: _types.INSTALL_STATES.SAVE_ARCHIVE_ENTRIES,
    onPostTransition: _steps.updateLatestExecutedState
  },
  save_archive_entries_from_assets_map: {
    onPreTransition: _steps.cleanupArchiveEntriesStep,
    onTransition: _steps.stepSaveArchiveEntries,
    nextState: _types.INSTALL_STATES.SAVE_KNOWLEDGE_BASE,
    onPostTransition: _steps.updateLatestExecutedState
  },
  save_knowledge_base: {
    onPreTransition: _steps.cleanupKnowledgeBaseStep,
    onTransition: _steps.stepSaveKnowledgeBase,
    nextState: _types.INSTALL_STATES.UPDATE_SO,
    onPostTransition: _steps.updateLatestExecutedState
  },
  update_so: {
    onPreTransition: _steps.cleanUpUnusedKibanaAssetsStep,
    onTransition: _steps.stepSaveSystemObject,
    nextState: 'end',
    onPostTransition: _steps.updateLatestExecutedState
  }
};

/*
 * _stateMachineInstallPackage installs packages using the generic state machine in ./state_machine
 * installStates is the data structure providing the state machine definition
 * Usually the install process starts with `create_restart_installation` and continues based on nextState parameter in the definition
 * The `onTransition` functions are the steps executed to go from one state to another, and all accept an `InstallContext` object as input parameter
 * After each transition `updateLatestExecutedState` is executed, it updates the executed state in the SO
 */
async function _stateMachineInstallPackage(context) {
  var _installedPkg$attribu, _installedPkg$attribu2;
  const {
    installedPkg,
    retryFromLastState,
    force
  } = context;
  const logger = _.appContextService.getLogger();
  let initialState = _types.INSTALL_STATES.CREATE_RESTART_INSTALLATION;
  const statesDefinition = context.useStreaming ? streamingStatesDefinition : regularStatesDefinition;

  // if retryFromLastState, restart install from last install state
  // if force is passed, the install should be executed from the beginning
  if (retryFromLastState && !force && installedPkg !== null && installedPkg !== void 0 && (_installedPkg$attribu = installedPkg.attributes) !== null && _installedPkg$attribu !== void 0 && (_installedPkg$attribu2 = _installedPkg$attribu.latest_executed_state) !== null && _installedPkg$attribu2 !== void 0 && _installedPkg$attribu2.name) {
    initialState = findNextState(installedPkg.attributes.latest_executed_state.name, statesDefinition);
    logger.debug(`Install with retryFromLastState option - Initial installation state: ${initialState}`);
    // we need to clean up latest_executed_state or it won't be refreshed
    await (0, _steps.cleanupLatestExecutedState)(context);
  }
  const installStates = {
    // inject initial state inside context
    context: {
      ...context,
      initialState
    },
    states: statesDefinition
  };
  try {
    const {
      installedKibanaAssetsRefs,
      esReferences
    } = await (0, _state_machine.handleState)(initialState, installStates, installStates.context);
    return [...(installedKibanaAssetsRefs ? installedKibanaAssetsRefs : []), ...(esReferences ? esReferences : [])];
  } catch (err) {
    const {
      packageInfo
    } = installStates.context.packageInstallContext;
    const {
      name: pkgName,
      version: pkgVersion
    } = packageInfo;
    if (_server.SavedObjectsErrorHelpers.isConflictError(err)) {
      throw new _errors.PackageSavedObjectConflictError(`Saved Object conflict encountered while installing ${pkgName || 'unknown'}-${pkgVersion || 'unknown'}. There may be a conflicting Saved Object saved to another Space. Original error: ${err.message}`);
    } else {
      throw err;
    }
  }
}
const findNextState = (latestExecutedState, states) => {
  return states[latestExecutedState].nextState;
};