"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IntegrationFieldsRepository = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _field_metadata = require("../../../../common/fields_metadata/models/field_metadata");
var _fields_metadata = require("../../../../common/fields_metadata");
var _hashed_cache = require("../../../../common/hashed_cache");
var _errors = require("../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class IntegrationFieldsRepository {
  constructor(integrationFieldsExtractor, integrationListExtractor) {
    (0, _defineProperty2.default)(this, "cache", void 0);
    (0, _defineProperty2.default)(this, "integrationsMap", void 0);
    (0, _defineProperty2.default)(this, "storeFieldsInCache", (cacheKey, extractedFieldsMetadata) => {
      const cachedIntegration = this.cache.get(cacheKey);
      if (!cachedIntegration) {
        this.cache.set(cacheKey, extractedFieldsMetadata);
      } else {
        this.cache.set(cacheKey, {
          ...cachedIntegration,
          ...extractedFieldsMetadata
        });
      }
    });
    (0, _defineProperty2.default)(this, "getCacheKey", ({
      integration,
      dataset
    }) => {
      const integrationDetails = this.integrationsMap.get(integration);
      if (integrationDetails) {
        return {
          dataset,
          integration,
          version: integrationDetails.version
        };
      }
      return {
        integration,
        dataset
      };
    });
    (0, _defineProperty2.default)(this, "mapExtractedFieldsToFieldMetadataTree", extractedFields => {
      const datasetGroups = Object.entries(extractedFields);
      return datasetGroups.reduce((integrationGroup, [datasetName, datasetGroup]) => {
        const datasetFieldsEntries = Object.entries(datasetGroup);
        integrationGroup[datasetName] = datasetFieldsEntries.reduce((datasetFields, [fieldName, field]) => {
          datasetFields[fieldName] = _field_metadata.FieldMetadata.create({
            ...field,
            source: 'integration'
          });
          return datasetFields;
        }, {});
        return integrationGroup;
      }, {});
    });
    (0, _defineProperty2.default)(this, "mapExtractedIntegrationListToMap", extractedIntegrations => {
      return new Map(extractedIntegrations.map(integration => [integration.name, integration]));
    });
    this.integrationFieldsExtractor = integrationFieldsExtractor;
    this.integrationListExtractor = integrationListExtractor;
    this.cache = new _hashed_cache.HashedCache();
    this.integrationsMap = new Map();
    this.extractIntegrationList();
  }
  async getByName(fieldName, params) {
    const {
      integration,
      dataset
    } = this.extractIntegrationFieldsSearchParams(fieldName, params);
    if (!integration || !this.integrationsMap.has(integration)) {
      return undefined;
    }
    let field = this.getCachedField(fieldName, {
      integration,
      dataset
    });
    if (!field) {
      try {
        await this.extractFields({
          integration,
          dataset
        });
      } catch (error) {
        throw new _errors.PackageNotFoundError(error.message);
      }
      field = this.getCachedField(fieldName, {
        integration,
        dataset
      });
    }
    return field;
  }
  static create({
    integrationFieldsExtractor,
    integrationListExtractor
  }) {
    return new IntegrationFieldsRepository(integrationFieldsExtractor, integrationListExtractor);
  }
  extractIntegrationFieldsSearchParams(fieldName, params) {
    var _params$integration, _params$dataset;
    const parts = fieldName.split('.');
    if (parts.length < 3) {
      return params;
    }
    const [extractedIntegration, extractedDataset] = parts;
    return {
      integration: (_params$integration = params.integration) !== null && _params$integration !== void 0 ? _params$integration : extractedIntegration,
      dataset: (_params$dataset = params.dataset) !== null && _params$dataset !== void 0 ? _params$dataset : [extractedIntegration, extractedDataset].join('.')
    };
  }
  async extractFields({
    integration,
    dataset
  }) {
    const cacheKey = this.getCacheKey({
      integration,
      dataset
    });
    const cachedIntegration = this.cache.get(cacheKey);
    if (cachedIntegration) {
      return undefined;
    }
    return this.integrationFieldsExtractor({
      integration,
      dataset
    }).then(this.mapExtractedFieldsToFieldMetadataTree).then(fieldMetadataTree => this.storeFieldsInCache(cacheKey, fieldMetadataTree));
  }
  extractIntegrationList() {
    void this.integrationListExtractor().then(this.mapExtractedIntegrationListToMap).then(integrationsMap => this.integrationsMap = integrationsMap);
  }
  getCachedField(fieldName, {
    integration,
    dataset
  }) {
    const cacheKey = this.getCacheKey({
      integration,
      dataset
    });
    const cachedIntegration = this.cache.get(cacheKey);
    const datasetName = dataset === _fields_metadata.ANY_DATASET ? null : dataset;

    // 1. Integration fields were never fetched
    if (!cachedIntegration) {
      return undefined;
    }

    // 2. Dataset is passed but was never fetched before
    if (datasetName && !Object.hasOwn(cachedIntegration, datasetName)) {
      return undefined;
    }

    // 3. Dataset is passed and it was previously fetched, should return the field
    if (datasetName && Object.hasOwn(cachedIntegration, datasetName)) {
      const targetDataset = cachedIntegration[datasetName];
      return targetDataset[fieldName];
    }

    // 4. Dataset is not passed, we attempt search on all stored datasets
    if (!datasetName) {
      // Merge all the available datasets into a unique field list. Overriding fields might occur in the process.
      const cachedDatasetsFields = Object.assign({}, ...Object.values(cachedIntegration));
      return cachedDatasetsFields[fieldName];
    }
  }
}
exports.IntegrationFieldsRepository = IntegrationFieldsRepository;