"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerLookupIndexRoutes = void 0;
var _configSchema = require("@kbn/config-schema");
var _esqlTypes = require("@kbn/esql-types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const registerLookupIndexRoutes = (router, {
  logger
}) => {
  router.post({
    path: `${_esqlTypes.LOOKUP_INDEX_UPDATE_ROUTE}/{indexName}`,
    validate: {
      params: _configSchema.schema.object({
        indexName: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        operations: _configSchema.schema.arrayOf(_configSchema.schema.any())
      })
    },
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    options: {
      description: 'Updates an index with bulk operations'
    }
  }, async (requestHandlerContext, request, response) => {
    try {
      const core = await requestHandlerContext.core;
      const client = core.elasticsearch.client.asCurrentUser;
      const indexName = request.params.indexName;
      const result = await client.bulk({
        index: indexName,
        operations: request.body.operations
      });
      return response.ok({
        body: result
      });
    } catch (error) {
      logger.get().debug(error);
      throw error;
    }
  });
  router.post({
    path: `${_esqlTypes.LOOKUP_INDEX_CREATE_ROUTE}/{indexName}`,
    validate: {
      params: _configSchema.schema.object({
        indexName: _configSchema.schema.string()
      })
    },
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    options: {
      description: 'Creates an index with lookup mode'
    }
  }, async (requestHandlerContext, request, response) => {
    try {
      const core = await requestHandlerContext.core;
      const client = core.elasticsearch.client.asCurrentUser;
      const indexName = request.params.indexName;
      const result = await client.indices.create({
        index: indexName,
        settings: {
          mode: 'lookup'
        }
      });
      return response.ok({
        body: result
      });
    } catch (error) {
      logger.get().debug(error);
      throw error;
    }
  });
  router.post({
    path: `${_esqlTypes.LOOKUP_INDEX_RECREATE_ROUTE}/{indexName}`,
    validate: {
      params: _configSchema.schema.object({
        indexName: _configSchema.schema.string()
      })
    },
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    options: {
      description: 'Recreates an index with lookup mode, used to reset the mapping of the index.'
    }
  }, async (requestHandlerContext, request, response) => {
    try {
      const core = await requestHandlerContext.core;
      const client = core.elasticsearch.client.asCurrentUser;
      const indexName = request.params.indexName;
      await client.indices.delete({
        index: indexName,
        ignore_unavailable: true
      });
      const result = await client.indices.create({
        index: indexName,
        settings: {
          mode: 'lookup'
        }
      });
      return response.ok({
        body: result
      });
    } catch (error) {
      logger.get().debug(error);
      throw error;
    }
  });
  router.get({
    path: _esqlTypes.LOOKUP_INDEX_PRIVILEGES_ROUTE,
    validate: {
      query: _configSchema.schema.object({
        indexName: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    },
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    options: {
      description: 'Checks if the user has privileges to create, read and write to an index'
    }
  }, async (requestHandlerContext, req, res) => {
    const {
      indexName
    } = req.query;
    const indices = indexName ? indexName.split(',') : [];
    try {
      const core = await requestHandlerContext.core;
      const esClient = core.elasticsearch.client.asCurrentUser;
      const {
        index: indexPrivileges = {}
      } = await esClient.security.hasPrivileges({
        index: [{
          names: ['*', ...indices],
          privileges: ['create_index', 'read', 'write', 'delete_index']
        }]
      });
      return res.ok({
        body: indexPrivileges
      });
    } catch (error) {
      logger.get().debug(error);
      throw error;
    }
  });
};
exports.registerLookupIndexRoutes = registerLookupIndexRoutes;