"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateVariableName = exports.updateQueryStringWithVariable = exports.getVariableTypeFromQuery = exports.getVariableSuggestion = exports.getVariableNamePrefix = exports.getRecurrentVariableName = exports.getQueryForFields = exports.flyoutStyles = exports.checkVariableExistence = exports.areValuesIntervalsValid = void 0;
var _esqlTypes = require("@kbn/esql-types");
var _esqlAst = require("@kbn/esql-ast");
var _react = require("@emotion/react");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function inKnownTimeInterval(timeIntervalUnit) {
  return _esqlAst.TIME_SPAN_UNITS.some(unit => unit === timeIntervalUnit.toLowerCase());
}
const getQueryPart = (queryString, cursorColumn, variable) => {
  const queryStringTillCursor = queryString.slice(0, cursorColumn);
  const lastChar = queryStringTillCursor.slice(-1);
  const secondLastChar = queryStringTillCursor.slice(-2, -1);
  if (lastChar === '?') {
    return [queryString.slice(0, cursorColumn - 2), variable, queryString.slice(cursorColumn - 1)].join('');
  } else if (secondLastChar === '?') {
    return [queryString.slice(0, cursorColumn - 2), variable, queryString.slice(cursorColumn - 1)].join('');
  }
  return [queryString.slice(0, cursorColumn - 1), variable, queryString.slice(cursorColumn - 1)].join('');
};
const updateQueryStringWithVariable = (queryString, variable, cursorPosition) => {
  var _cursorPosition$colum, _cursorPosition$lineN;
  const cursorColumn = (_cursorPosition$colum = cursorPosition === null || cursorPosition === void 0 ? void 0 : cursorPosition.column) !== null && _cursorPosition$colum !== void 0 ? _cursorPosition$colum : 0;
  const cursorLine = (_cursorPosition$lineN = cursorPosition === null || cursorPosition === void 0 ? void 0 : cursorPosition.lineNumber) !== null && _cursorPosition$lineN !== void 0 ? _cursorPosition$lineN : 0;
  const lines = queryString.split('\n');
  if (lines.length > 1) {
    const queryArray = queryString.split('\n');
    const queryPartToBeUpdated = queryArray[cursorLine - 1];
    const queryWithVariable = getQueryPart(queryPartToBeUpdated, cursorColumn, variable);
    queryArray[cursorLine - 1] = queryWithVariable;
    return queryArray.join('\n');
  }
  return [getQueryPart(queryString, cursorColumn, variable)].join('');
};
exports.updateQueryStringWithVariable = updateQueryStringWithVariable;
const getQueryForFields = (queryString, cursorPosition) => {
  var _cursorPosition$colum2, _cursorPosition$lineN2;
  const cursorColumn = (_cursorPosition$colum2 = cursorPosition === null || cursorPosition === void 0 ? void 0 : cursorPosition.column) !== null && _cursorPosition$colum2 !== void 0 ? _cursorPosition$colum2 : 0;
  const cursorLine = (_cursorPosition$lineN2 = cursorPosition === null || cursorPosition === void 0 ? void 0 : cursorPosition.lineNumber) !== null && _cursorPosition$lineN2 !== void 0 ? _cursorPosition$lineN2 : 0;
  const lines = queryString.split('\n');
  if (lines.length > 1) {
    const queryArray = queryString.split('\n');
    const lineToBeUpdated = cursorLine - 1;
    return queryArray.slice(0, lineToBeUpdated).join('\n');
  }
  const queryBefore = queryString.slice(0, cursorColumn - 1);
  const pipes = queryBefore.split('|');
  return pipes.slice(0, pipes.length - 1).join('|');
};
exports.getQueryForFields = getQueryForFields;
const areValuesIntervalsValid = values => {
  return values.every(value => {
    // remove digits and empty spaces from the string to get the unit
    const unit = value.replace(/[0-9]/g, '').replace(/\s/g, '');
    return inKnownTimeInterval(unit);
  });
};
exports.areValuesIntervalsValid = areValuesIntervalsValid;
const getVariableSuggestion = variableType => {
  switch (variableType) {
    case _esqlTypes.ESQLVariableType.FIELDS:
      return 'field';
    case _esqlTypes.ESQLVariableType.FUNCTIONS:
      return 'function';
    case _esqlTypes.ESQLVariableType.TIME_LITERAL:
      return 'interval';
    case _esqlTypes.ESQLVariableType.MULTI_VALUES:
      return 'values';
    default:
      return 'variable';
  }
};
exports.getVariableSuggestion = getVariableSuggestion;
const getRecurrentVariableName = (name, existingNames) => {
  let newName = name;
  let i = 1;
  while (existingNames.has(newName)) {
    newName = `${name}${i}`;
    i++;
  }
  return newName;
};
exports.getRecurrentVariableName = getRecurrentVariableName;
const flyoutStyles = exports.flyoutStyles = (0, _react.css)({
  '.euiFlyoutBody__overflow': {
    WebkitMaskImage: 'none',
    paddingLeft: 'inherit',
    marginLeft: 'inherit',
    transform: 'initial'
  },
  '.euiFlyoutBody__overflowContent': {
    blockSize: '100%'
  }
});
const validateVariableName = (variableName, prefix) => {
  let text = variableName
  // variable name can only contain letters, numbers, underscores and questionmarks
  .replace(/[^a-zA-Z0-9_?]/g, '');
  if (!text.startsWith('?')) {
    text = `?${text}`;
  }
  const match = text.match(/^(\?*)/);
  const leadingQuestionMarksCount = match ? match[0].length : 0;
  if (leadingQuestionMarksCount > 2) {
    text = '??'.concat(text.substring(leadingQuestionMarksCount));
  }

  // Remove unnecessary leading underscores
  if (text.charAt(prefix.length) === '_') {
    text = `${prefix}${text.substring(prefix.length + 1)}`;
  }
  return text;
};
exports.validateVariableName = validateVariableName;
const getVariableTypeFromQuery = (str, variableType) => {
  const match = str.match(/^(\?*)/);
  const leadingQuestionMarksCount = match ? match[0].length : 0;
  if (leadingQuestionMarksCount === 2 && variableType !== _esqlTypes.ESQLVariableType.FIELDS && variableType !== _esqlTypes.ESQLVariableType.FUNCTIONS) {
    return _esqlTypes.ESQLVariableType.FIELDS;
  }
  if (leadingQuestionMarksCount === 1 && variableType !== _esqlTypes.ESQLVariableType.TIME_LITERAL && variableType !== _esqlTypes.ESQLVariableType.VALUES && variableType !== _esqlTypes.ESQLVariableType.MULTI_VALUES) {
    return _esqlTypes.ESQLVariableType.VALUES;
  }
  return variableType;
};
exports.getVariableTypeFromQuery = getVariableTypeFromQuery;
const getVariableNamePrefix = type => {
  switch (type) {
    case _esqlTypes.ESQLVariableType.FIELDS:
    case _esqlTypes.ESQLVariableType.FUNCTIONS:
      return _esqlTypes.VariableNamePrefix.IDENTIFIER;
    case _esqlTypes.ESQLVariableType.VALUES:
    case _esqlTypes.ESQLVariableType.TIME_LITERAL:
    case _esqlTypes.ESQLVariableType.MULTI_VALUES:
    default:
      return _esqlTypes.VariableNamePrefix.VALUE;
  }
};
exports.getVariableNamePrefix = getVariableNamePrefix;
const checkVariableExistence = (esqlVariables, variableName) => {
  const variableNameWithoutQuestionmark = variableName.replace(/^\?+/, '');
  const match = variableName.match(/^(\?*)/);
  const leadingQuestionMarksCount = match ? match[0].length : 0;
  return esqlVariables.some(variable => {
    const prefix = getVariableNamePrefix(variable.type);
    if (leadingQuestionMarksCount === 2) {
      if (prefix === _esqlTypes.VariableNamePrefix.IDENTIFIER) {
        return variable.key === variableNameWithoutQuestionmark;
      }
      return false;
    }
    if (leadingQuestionMarksCount === 1) {
      if (prefix === _esqlTypes.VariableNamePrefix.VALUE) {
        return variable.key === variableNameWithoutQuestionmark;
      }
      return false;
    }
    return false;
  });
};
exports.checkVariableExistence = checkVariableExistence;