"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.extractDissectPattern = extractDissectPattern;
var _sanitize_bracket_delimiters = require("./sanitize_bracket_delimiters");
var _find_delimiter_sequences = require("./find_delimiter_sequences");
var _build_delimiter_tree = require("./build_delimiter_tree");
var _extract_fields = require("./extract_fields");
var _normalize_field_boundaries = require("./normalize_field_boundaries");
var _normalize_whitespace = require("./normalize_whitespace");
var _detect_modifiers = require("./detect_modifiers");
var _generate_ast = require("./generate_ast");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * WARNING: DO NOT RUN THIS FUNCTION ON THE MAIN THREAD
 *
 * Extracts a Dissect pattern from an array of log messages by analyzing
 * common delimiters and structure.
 *
 * This function performs multiple passes to identify consistent delimiter patterns
 * and normalize the data into a structured format. It is computationally intensive
 * (O(n × m²) complexity) and should not be run on the main thread.
 *
 * Steps:
 * 1. Find common delimiter sequences that appear in all messages.
 * 2. Build an ordered delimiter tree by position.
 * 3. Extract variable regions between delimiters as fields.
 * 4. Detect modifiers (right padding, skip fields).
 * 5. Generate the final Dissect pattern string.
 *
 * @param messages - Array of log message strings to analyze
 * @returns DissectPattern object with pattern string and field metadata
 */
function extractDissectPattern(messages) {
  if (!messages.length) {
    return {
      ast: {
        nodes: []
      },
      fields: []
    };
  }

  // Step 0: Normalize whitespace to handle varying amounts of spaces/tabs
  // This ensures delimiters align at consistent positions across all messages
  const normalizedMessages = (0, _normalize_whitespace.normalizeWhitespace)(messages);
  const normalizedStrings = normalizedMessages.map(nm => nm.normalized);

  // Step 1: Find common delimiter sequences (on normalized messages)
  const delimiters = (0, _find_delimiter_sequences.findDelimiterSequences)(normalizedStrings);

  // Step 2: Build ordered delimiter tree (on normalized messages)
  let delimiterTree = (0, _build_delimiter_tree.buildDelimiterTree)(normalizedStrings, delimiters);

  // Step 3: Extract fields between delimiters (on normalized messages)
  let fields = (0, _extract_fields.extractFields)(normalizedStrings, delimiterTree);

  // Step 3.5: Normalize field boundaries (move trailing non-alphanumeric chars to delimiters)
  (0, _normalize_field_boundaries.normalizeFieldBoundaries)(fields, delimiterTree);

  // Step 3.6: Bracket sanitization (mismatched/unmatched bracket characters removed from literals).
  // This mirrors the previous in-place sanitization but is now executed after field boundary normalization.
  delimiterTree = (0, _sanitize_bracket_delimiters.sanitizeBracketDelimiters)(delimiterTree);

  // Recompute fields to reflect updated delimiter literals.
  fields = (0, _extract_fields.extractFields)(normalizedStrings, delimiterTree);

  // Step 4: Detect modifiers for each field
  fields.forEach((field, index) => {
    field.modifiers = (0, _detect_modifiers.detectModifiers)(field);

    // Apply right-padding modifier based on delimiter positions
    // Find the delimiter that comes after this field by comparing positions
    // Use MIN length since we want to find the delimiter that appears right after
    // the shortest value of this field (where the field ends in all messages)
    const minFieldLength = Math.min(...field.values.map(v => v.length));
    const fieldEndMin = field.position + minFieldLength;
    const nextDelimiter = delimiterTree.find(d => Math.min(...d.positions) >= fieldEndMin);
    if (nextDelimiter && nextDelimiter.positions.length > 0) {
      // Check if this delimiter appears at varying positions or where whitespace was collapsed
      if ((0, _normalize_whitespace.needsRightPadding)(nextDelimiter.positions, normalizedMessages, nextDelimiter.literal)) {
        field.modifiers.rightPadding = true;
      }
    }
  });

  // Step 5: Generate AST
  const ast = (0, _generate_ast.generateAST)(delimiterTree, fields);
  return {
    ast,
    fields
  };
}