"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createTabsStorageManager = exports.TABS_LOCAL_STORAGE_KEY = exports.RECENTLY_CLOSED_TABS_LIMIT = void 0;
var _lodash = require("lodash");
var _public = require("@kbn/kibana-utils-plugin/public");
var _constants = require("../../../../common/constants");
var _utils = require("./redux/utils");
var _redux = require("./redux");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const TABS_LOCAL_STORAGE_KEY = exports.TABS_LOCAL_STORAGE_KEY = 'discover.tabs';
const RECENTLY_CLOSED_TABS_LIMIT = exports.RECENTLY_CLOSED_TABS_LIMIT = 50;
const defaultTabsStateInLocalStorage = {
  userId: '',
  spaceId: '',
  discoverSessionId: undefined,
  openTabs: [],
  closedTabs: []
};
const createTabsStorageManager = ({
  urlStateStorage,
  storage,
  enabled
}) => {
  const urlStateContainer = (0, _public.createStateContainer)({});
  const sessionInfo = {
    userId: '',
    spaceId: ''
  };

  // Used to avoid triggering onChanged during programmatic tab ID URL updates
  let isPushingTabIdToUrl = false;
  const startUrlSync = ({
    onChanged // can be called when selectedTabId changes in URL to trigger app state change if needed
  }) => {
    if (!enabled) {
      return () => {
        // do nothing
      };
    }
    const {
      start,
      stop
    } = (0, _public.syncState)({
      stateStorage: urlStateStorage,
      stateContainer: {
        ...urlStateContainer,
        set: state => {
          if (state) {
            // syncState utils requires to handle incoming "null" value
            urlStateContainer.set(state);
          }
        }
      },
      storageKey: _constants.TAB_STATE_URL_KEY
    });
    const listener = onChanged ? urlStateContainer.state$.subscribe(state => {
      if (!isPushingTabIdToUrl) {
        onChanged(state);
      }
    }) : null;
    start();
    return () => {
      listener === null || listener === void 0 ? void 0 : listener.unsubscribe();
      stop();
    };
  };
  const getTabsStateFromURL = () => {
    return urlStateStorage.get(_constants.TAB_STATE_URL_KEY);
  };
  const pushSelectedTabIdToUrl = async (selectedTabId, {
    replace = false
  } = {}) => {
    const nextState = {
      tabId: selectedTabId
    };
    const previousState = getTabsStateFromURL();
    // If the previous tab was a "new" (unsaved) tab, we replace the URL state instead of pushing a new history entry.
    // This prevents cluttering the browser history with intermediate "new tab" states that are not meaningful to the user.
    const shouldReplace = replace || (previousState === null || previousState === void 0 ? void 0 : previousState.tabId) === _constants.NEW_TAB_ID;
    try {
      isPushingTabIdToUrl = true;
      await urlStateStorage.set(_constants.TAB_STATE_URL_KEY, nextState, {
        replace: shouldReplace
      });
    } finally {
      isPushingTabIdToUrl = false;
    }
  };
  const toTabStateInStorage = (tabState, getInternalState) => {
    const getInternalStateForTabWithoutRuntimeState = tabId => (getInternalState === null || getInternalState === void 0 ? void 0 : getInternalState(tabId)) || tabState.initialInternalState;
    return {
      id: tabState.id,
      label: tabState.label,
      internalState: getInternalStateForTabWithoutRuntimeState(tabState.id),
      appState: tabState.appState,
      globalState: tabState.globalState
    };
  };
  const toRecentlyClosedTabStateInStorage = tabState => {
    const state = toTabStateInStorage(tabState, undefined);
    return {
      ...state,
      closedAt: tabState.closedAt
    };
  };
  const getDefinedStateOnly = state => {
    if (!state || !Object.keys(state).length) {
      return undefined;
    }
    return state;
  };
  const toTabState = (tabStateInStorage, defaultTabState) => {
    const internalState = getDefinedStateOnly(tabStateInStorage.internalState);
    const appState = getDefinedStateOnly(tabStateInStorage.appState);
    const globalState = getDefinedStateOnly(tabStateInStorage.globalState || defaultTabState.globalState);
    const controlGroupState = internalState !== null && internalState !== void 0 && internalState.controlGroupJson ? (0, _utils.parseControlGroupJson)(internalState.controlGroupJson) : undefined;
    const esqlVariables = controlGroupState ? (0, _utils.extractEsqlVariables)(controlGroupState) : defaultTabState.esqlVariables;
    return {
      ...defaultTabState,
      ...(0, _lodash.pick)(tabStateInStorage, 'id', 'label'),
      initialInternalState: internalState,
      appState: appState || {},
      globalState: globalState || {},
      esqlVariables
    };
  };
  const toRecentlyClosedTabState = (tabStateInStorage, defaultTabState) => ({
    ...toTabState(tabStateInStorage, defaultTabState),
    closedAt: tabStateInStorage.closedAt
  });
  const readFromLocalStorage = () => {
    const storedTabsState = storage.get(TABS_LOCAL_STORAGE_KEY);
    return {
      userId: (storedTabsState === null || storedTabsState === void 0 ? void 0 : storedTabsState.userId) || '',
      spaceId: (storedTabsState === null || storedTabsState === void 0 ? void 0 : storedTabsState.spaceId) || '',
      discoverSessionId: (storedTabsState === null || storedTabsState === void 0 ? void 0 : storedTabsState.discoverSessionId) || undefined,
      openTabs: (storedTabsState === null || storedTabsState === void 0 ? void 0 : storedTabsState.openTabs) || [],
      closedTabs: (storedTabsState === null || storedTabsState === void 0 ? void 0 : storedTabsState.closedTabs) || []
    };
  };
  const getNRecentlyClosedTabs = ({
    previousOpenTabs,
    previousRecentlyClosedTabs,
    nextOpenTabs,
    justRemovedTabs
  }) => {
    var _latestNRecentlyClose;
    const removedTabs = justRemovedTabs !== null && justRemovedTabs !== void 0 ? justRemovedTabs : (0, _lodash.differenceBy)(previousOpenTabs, nextOpenTabs, 'id');
    const closedAt = Date.now();
    const newRecentlyClosedTabs = removedTabs.map(tab => ({
      ...tab,
      closedAt
    }));
    const newSortedRecentlyClosedTabs = (0, _lodash.orderBy)((0, _lodash.uniqBy)([...newRecentlyClosedTabs, ...previousRecentlyClosedTabs], 'id'),
    // prevent duplicates
    'closedAt', 'desc');
    const latestNRecentlyClosedTabs = newSortedRecentlyClosedTabs.slice(0, RECENTLY_CLOSED_TABS_LIMIT);
    const recentClosedAt = (_latestNRecentlyClose = latestNRecentlyClosedTabs[latestNRecentlyClosedTabs.length - 1]) === null || _latestNRecentlyClose === void 0 ? void 0 : _latestNRecentlyClose.closedAt;
    if (recentClosedAt) {
      // keep other recently closed tabs from the same time point when they were closed
      for (let i = RECENTLY_CLOSED_TABS_LIMIT; i < newSortedRecentlyClosedTabs.length; i++) {
        if (newSortedRecentlyClosedTabs[i].closedAt === recentClosedAt) {
          latestNRecentlyClosedTabs.push(newSortedRecentlyClosedTabs[i]);
        } else {
          break;
        }
      }
    }
    return latestNRecentlyClosedTabs;
  };
  const persistLocally = async ({
    allTabs,
    recentlyClosedTabs
  }, getInternalState, discoverSessionId) => {
    if (!enabled) {
      return;
    }
    const openTabs = allTabs.map(tab => toTabStateInStorage(tab, getInternalState));
    const closedTabs = recentlyClosedTabs.map(tab => toRecentlyClosedTabStateInStorage(tab));
    const nextTabsInStorage = {
      userId: sessionInfo.userId,
      spaceId: sessionInfo.spaceId,
      discoverSessionId,
      openTabs,
      closedTabs // wil be used for "Recently closed tabs" feature
    };
    storage.set(TABS_LOCAL_STORAGE_KEY, nextTabsInStorage);
  };
  const updateTabStateLocally = (tabId, tabStatePartial) => {
    if (!enabled) {
      return;
    }
    let hasModifications = false;
    const storedTabsState = readFromLocalStorage();
    const updatedTabsState = {
      ...storedTabsState,
      openTabs: storedTabsState.openTabs.map(tab => {
        if (tab.id === tabId) {
          hasModifications = true;
          return {
            ...tab,
            internalState: tabStatePartial.internalState,
            appState: tabStatePartial.appState,
            globalState: tabStatePartial.globalState
          };
        }
        return tab;
      })
    };
    if (hasModifications) {
      storage.set(TABS_LOCAL_STORAGE_KEY, updatedTabsState);
    }
  };
  const loadLocally = ({
    userId,
    spaceId,
    persistedDiscoverSession,
    shouldClearAllTabs,
    defaultTabState
  }) => {
    const tabsStateFromURL = getTabsStateFromURL();
    const selectedTabId = enabled ? shouldClearAllTabs ? undefined : tabsStateFromURL === null || tabsStateFromURL === void 0 ? void 0 : tabsStateFromURL.tabId : undefined;
    let storedTabsState = enabled ? readFromLocalStorage() : defaultTabsStateInLocalStorage;
    if (storedTabsState.userId !== userId || storedTabsState.spaceId !== spaceId) {
      // if the userId or spaceId has changed, don't read from the local storage
      storedTabsState = {
        ...defaultTabsStateInLocalStorage,
        userId,
        spaceId
      };
    }
    sessionInfo.userId = userId;
    sessionInfo.spaceId = spaceId;
    const persistedTabs = persistedDiscoverSession === null || persistedDiscoverSession === void 0 ? void 0 : persistedDiscoverSession.tabs.map(tab => (0, _redux.fromSavedObjectTabToTabState)({
      tab
    }));
    const previousOpenTabs = storedTabsState.openTabs.map(tab => toTabState(tab, defaultTabState));
    let openTabs = shouldClearAllTabs ? [] : previousOpenTabs;
    if ((persistedDiscoverSession === null || persistedDiscoverSession === void 0 ? void 0 : persistedDiscoverSession.id) !== storedTabsState.discoverSessionId) {
      // if the discover session has changed, use the tabs from the session
      openTabs = persistedTabs !== null && persistedTabs !== void 0 ? persistedTabs : [];
    }
    const closedTabs = storedTabsState.closedTabs.map(tab => toRecentlyClosedTabState(tab, defaultTabState));

    // restore previously opened tabs
    if (enabled) {
      // try to preselect one of the previously opened tabs
      if (selectedTabId && selectedTabId !== _constants.NEW_TAB_ID && openTabs.find(tab => tab.id === selectedTabId)) {
        return {
          allTabs: openTabs,
          selectedTabId,
          recentlyClosedTabs: getNRecentlyClosedTabs({
            previousOpenTabs,
            previousRecentlyClosedTabs: closedTabs,
            nextOpenTabs: openTabs
          })
        };
      }
      if (
      // append a new tab if requested via URL
      selectedTabId === _constants.NEW_TAB_ID ||
      // or append a new tab to the persisted session if could not find it by the selected tab id above
      selectedTabId && tabsStateFromURL !== null && tabsStateFromURL !== void 0 && tabsStateFromURL.tabLabel && persistedDiscoverSession) {
        const newTab = {
          ...defaultTabState,
          ...(0, _utils.createTabItem)(openTabs)
        };
        if (tabsStateFromURL !== null && tabsStateFromURL !== void 0 && tabsStateFromURL.tabLabel) {
          newTab.label = tabsStateFromURL.tabLabel;
        }
        const allTabsWithNewTab = [...openTabs, newTab];
        return {
          allTabs: allTabsWithNewTab,
          selectedTabId: newTab.id,
          recentlyClosedTabs: getNRecentlyClosedTabs({
            previousOpenTabs,
            previousRecentlyClosedTabs: closedTabs,
            nextOpenTabs: allTabsWithNewTab
          })
        };
      }

      // otherwise try to reopen some of the previously closed tabs
      if (selectedTabId && !persistedDiscoverSession && !(tabsStateFromURL !== null && tabsStateFromURL !== void 0 && tabsStateFromURL.tabLabel)) {
        const storedClosedTab = storedTabsState.closedTabs.find(tab => tab.id === selectedTabId);
        if (storedClosedTab) {
          // restore previously closed tabs, for example when only the default tab was shown
          const restoredTabs = storedTabsState.closedTabs.filter(tab => tab.closedAt === storedClosedTab.closedAt).map(tab => toTabState(tab, defaultTabState));
          return {
            allTabs: restoredTabs,
            selectedTabId,
            recentlyClosedTabs: getNRecentlyClosedTabs({
              previousOpenTabs,
              previousRecentlyClosedTabs: closedTabs,
              nextOpenTabs: restoredTabs
            })
          };
        }
      }
    }

    // otherwise open the first tab from the Discover Session SO or a new default tab as a fallback
    const newDefaultTab = {
      ...defaultTabState,
      ...(0, _utils.createTabItem)([])
    };
    if (enabled && tabsStateFromURL !== null && tabsStateFromURL !== void 0 && tabsStateFromURL.tabLabel) {
      newDefaultTab.label = tabsStateFromURL.tabLabel;
    }
    let allTabs = [newDefaultTab];
    let selectedTab = newDefaultTab;
    if (persistedTabs !== null && persistedTabs !== void 0 && persistedTabs.length) {
      allTabs = persistedTabs;
      selectedTab = persistedTabs[0];
    }
    return {
      allTabs,
      selectedTabId: selectedTab.id,
      recentlyClosedTabs: getNRecentlyClosedTabs({
        previousOpenTabs,
        previousRecentlyClosedTabs: closedTabs,
        nextOpenTabs: allTabs
      })
    };
  };
  return {
    startUrlSync,
    pushSelectedTabIdToUrl,
    persistLocally,
    updateTabStateLocally,
    loadLocally,
    getNRecentlyClosedTabs
  };
};
exports.createTabsStorageManager = createTabsStorageManager;