"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createProjectFetcher = createProjectFetcher;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const MAX_RETRIES = 2;
const RETRY_DELAY_MS = 1000;
/**
 * Creates project fetcher with caching and retry logic
 */
function createProjectFetcher(http, logger) {
  let fetchPromise = null;
  let cachedData = null;
  async function fetchProjectsWithRetry() {
    let lastError = new Error('');
    for (let attempt = 0; attempt <= MAX_RETRIES; attempt++) {
      try {
        const response = await http.get('/internal/cps/projects_tags');
        const originValues = response.origin ? Object.values(response.origin) : [];
        return {
          origin: originValues.length > 0 ? originValues[0] : null,
          linkedProjects: response.linked_projects ? Object.values(response.linked_projects).sort((a, b) => a._alias.localeCompare(b._alias)) : []
        };
      } catch (error) {
        lastError = error instanceof Error ? error : new Error(String(error));
        logger.error(`Failed to fetch projects (attempt ${attempt + 1}/${MAX_RETRIES + 1})`, {
          error
        });
        if (attempt < MAX_RETRIES) {
          await new Promise(resolve => setTimeout(resolve, RETRY_DELAY_MS * Math.pow(2, attempt)));
        }
      }
    }
    throw lastError;
  }
  async function doFetch() {
    if (fetchPromise) {
      return fetchPromise;
    }
    fetchPromise = fetchProjectsWithRetry().then(projectsData => {
      cachedData = projectsData;
      return projectsData;
    }).finally(() => {
      fetchPromise = null;
    });
    return fetchPromise;
  }
  return {
    /**
     * Fetches projects from the server with caching and retry logic.
     * Returns cached data if already loaded. If a fetch is already in progress, returns the existing promise.
     */
    fetchProjects: async () => {
      if (cachedData) {
        return cachedData;
      }
      return doFetch();
    },
    /**
     * Forces a refresh of projects from the server, bypassing the cache.
     */
    refresh: async () => {
      return doFetch();
    }
  };
}