"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createDownstreamDependenciesTool = createDownstreamDependenciesTool;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _tool_schemas = require("./utils/tool_schemas");
var _build_apm_tool_resources = require("./utils/build_apm_tool_resources");
var _get_apm_tool_availability = require("./utils/get_apm_tool_availability");
var _get_apm_downstream_dependencies = require("../routes/assistant_functions/get_apm_downstream_dependencies");
var _agent_tool_ids = require("../../common/observability_agent/agent_tool_ids");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getDownstreamDependenciesToolSchema = _zod.z.object({
  ..._tool_schemas.timeRangeSchema.shape,
  serviceName: _zod.z.string().min(1).describe('The name of the service'),
  serviceEnvironment: _zod.z.string().optional().describe('The environment that the service is running in. Leave empty to query for all environments.')
});
function createDownstreamDependenciesTool({
  core,
  plugins,
  logger
}) {
  const toolDefinition = {
    id: _agent_tool_ids.OBSERVABILITY_GET_DOWNSTREAM_DEPENDENCIES_TOOL_ID,
    type: _onechatCommon.ToolType.builtin,
    description: 'Get downstream dependencies (services or uninstrumented backends) for a given service and time range.',
    schema: getDownstreamDependenciesToolSchema,
    tags: ['observability', 'apm', 'dependencies'],
    availability: {
      cacheMode: 'space',
      handler: async ({
        request
      }) => {
        return (0, _get_apm_tool_availability.getApmToolAvailability)({
          core,
          plugins,
          request,
          logger
        });
      }
    },
    handler: async (args, context) => {
      const {
        request,
        esClient,
        logger: scopedLogger
      } = context;
      try {
        const {
          apmEventClient,
          randomSampler
        } = await (0, _build_apm_tool_resources.buildApmToolResources)({
          core,
          plugins,
          request,
          esClient,
          logger: scopedLogger
        });
        const result = await (0, _get_apm_downstream_dependencies.getApmDownstreamDependencies)({
          arguments: args,
          apmEventClient,
          randomSampler
        });
        return {
          results: [{
            type: _tool_result.ToolResultType.other,
            data: {
              dependencies: result
            }
          }]
        };
      } catch (error) {
        logger.error(`Error getting APM downstream dependencies: ${error.message}`);
        logger.debug(error);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: `Failed to fetch downstream dependencies: ${error.message}`,
              stack: error.stack
            }
          }]
        };
      }
    }
  };
  return toolDefinition;
}