"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPropertiesAndSchema = getPropertiesAndSchema;
exports.buildSchema = buildSchema;
var tslib_1 = require("tslib");
var moment_1 = tslib_1.__importDefault(require("moment"));
// https://help.fullstory.com/hc/en-us/articles/360020623234#reserved-properties
var FULLSTORY_RESERVED_PROPERTIES = [
    'uid',
    'displayName',
    'email',
    'acctId',
    'website',
    // https://developer.fullstory.com/page-variables
    'pageName',
];
function getPropertiesAndSchema(context) {
    // Clean the properties by removing undefined at all levels
    var properties = removeUndefined(context);
    var schema = buildSchema(properties);
    return { properties: properties, schema: { properties: schema } };
}
function removeUndefined(context) {
    return Object.fromEntries(Object.entries(context)
        // Discard any undefined values
        .map(function (_a) {
        var _b = tslib_1.__read(_a, 2), key = _b[0], value = _b[1];
        return Array.isArray(value) ? [key, value.filter(function (v) { return typeof v !== 'undefined'; })] : [key, value];
    })
        .filter(function (_a) {
        var _b = tslib_1.__read(_a, 2), value = _b[1];
        return typeof value !== 'undefined' && (!Array.isArray(value) || value.length > 0);
    })
        .map(function (_a) {
        var _b = tslib_1.__read(_a, 2), key = _b[0], value = _b[1];
        if (isRecord(value)) {
            return [key, removeUndefined(value)];
        }
        return [key, value];
    }));
}
/**
 * Build the schema as per FullStory requirements https://developer.fullstory.com/browser/custom-properties/
 * @param context
 */
function buildSchema(context) {
    return Object.fromEntries(Object.entries(context)
        // Discard any undefined values
        .map(function (_a) {
        var _b = tslib_1.__read(_a, 2), key = _b[0], value = _b[1];
        return Array.isArray(value) ? [key, value.filter(function (v) { return typeof v !== 'undefined'; })] : [key, value];
    })
        // Discard reserved properties (no need to define them in the schema)
        .filter(function (_a) {
        var _b = tslib_1.__read(_a, 1), key = _b[0];
        return !FULLSTORY_RESERVED_PROPERTIES.includes(key);
    })
        .filter(function (_a) {
        var _b = tslib_1.__read(_a, 2), value = _b[1];
        return typeof value !== 'undefined' && (!Array.isArray(value) || value.length > 0);
    })
        // Infer the type according to the FullStory specs
        .map(function (_a) {
        var _b = tslib_1.__read(_a, 2), key = _b[0], value = _b[1];
        if (isRecord(value)) {
            return [key, buildSchema(value)];
        }
        var valueType = getFullStoryType(value);
        return [key, valueType];
    })
        .filter(function (_a) {
        var _b = tslib_1.__read(_a, 2), value = _b[1];
        return typeof value !== 'undefined';
    }));
}
function getFullStoryType(value) {
    // For arrays, make the decision based on the first element
    var isArray = Array.isArray(value);
    var v = isArray ? value[0] : value;
    var type;
    switch (typeof v) {
        case 'string':
            type = (0, moment_1.default)(v, moment_1.default.ISO_8601, true).isValid() ? 'date' : 'str';
            break;
        case 'number':
            type = Number.isInteger(v) ? 'int' : 'real';
            break;
        case 'boolean':
            type = 'bool';
            break;
        case 'object':
            if (isDate(v)) {
                type = 'date';
                break;
            }
        default:
            throw new Error("Unsupported type: ".concat(typeof v));
    }
    // convert to plural form for arrays
    return isArray ? "".concat(type, "s") : type;
}
function isRecord(value) {
    return typeof value === 'object' && value !== null && !Array.isArray(value) && !isDate(value);
}
function isDate(value) {
    return value instanceof Date;
}
