/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

package org.elasticsearch.action.admin.cluster.snapshots.features;

import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionType;
import org.elasticsearch.action.support.ActionFilters;
import org.elasticsearch.action.support.master.TransportMasterNodeAction;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.block.ClusterBlockException;
import org.elasticsearch.cluster.block.ClusterBlockLevel;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.util.concurrent.EsExecutors;
import org.elasticsearch.indices.SystemIndices;
import org.elasticsearch.injection.guice.Inject;
import org.elasticsearch.tasks.Task;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.TransportService;

public class TransportSnapshottableFeaturesAction extends TransportMasterNodeAction<
    GetSnapshottableFeaturesRequest,
    GetSnapshottableFeaturesResponse> {

    public static final ActionType<GetSnapshottableFeaturesResponse> TYPE = new ActionType<>("cluster:admin/features/get");
    private final SystemIndices systemIndices;

    @Inject
    public TransportSnapshottableFeaturesAction(
        TransportService transportService,
        ClusterService clusterService,
        ThreadPool threadPool,
        ActionFilters actionFilters,
        SystemIndices systemIndices
    ) {
        super(
            TYPE.name(),
            transportService,
            clusterService,
            threadPool,
            actionFilters,
            GetSnapshottableFeaturesRequest::new,
            GetSnapshottableFeaturesResponse::new,
            EsExecutors.DIRECT_EXECUTOR_SERVICE
        );
        this.systemIndices = systemIndices;
    }

    @Override
    protected void masterOperation(
        Task task,
        GetSnapshottableFeaturesRequest request,
        ClusterState state,
        ActionListener<GetSnapshottableFeaturesResponse> listener
    ) throws Exception {
        listener.onResponse(
            new GetSnapshottableFeaturesResponse(
                systemIndices.getFeatures()
                    .stream()
                    .map(feature -> new GetSnapshottableFeaturesResponse.SnapshottableFeature(feature.getName(), feature.getDescription()))
                    .toList()
            )
        );
    }

    @Override
    protected ClusterBlockException checkBlock(GetSnapshottableFeaturesRequest request, ClusterState state) {
        return state.blocks().globalBlockedException(ClusterBlockLevel.METADATA_READ);
    }
}
