/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index.store;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.store.IndexOutput;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.lucene.store.FilterIndexOutput;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.index.store.Store;
import org.elasticsearch.index.store.StoreFileMetadata;

final class VerifyingIndexOutput
extends FilterIndexOutput {
    private static final int CHECKSUM_LENGTH = 8;
    private final StoreFileMetadata metadata;
    private long writtenBytes;
    private final long checksumPosition;
    private final ByteBuffer computedChecksum = ByteBuffer.allocate(8);
    private final ByteBuffer footerChecksum = ByteBuffer.allocate(8);

    VerifyingIndexOutput(StoreFileMetadata metadata, IndexOutput out) throws IOException {
        super("VerifyingIndexOutput(out=" + out.toString() + ")", out);
        this.metadata = metadata;
        this.checksumPosition = metadata.length() - 8L;
        this.computedChecksum.order(ByteOrder.BIG_ENDIAN);
        this.footerChecksum.order(ByteOrder.BIG_ENDIAN);
        if (this.writtenBytes == this.checksumPosition) {
            this.computeChecksum();
        }
    }

    @Override
    public void writeByte(byte b) throws IOException {
        if (this.checksumPosition <= this.writtenBytes && this.writtenBytes < this.checksumPosition + 8L) {
            this.footerChecksum.put(b);
        }
        this.out.writeByte(b);
        ++this.writtenBytes;
        if (this.writtenBytes == this.checksumPosition) {
            this.computeChecksum();
        }
    }

    @Override
    public void writeBytes(byte[] b, int offset, int length) throws IOException {
        assert (0 < length);
        if (this.writtenBytes < this.checksumPosition) {
            int lengthToCopy = Math.toIntExact(Math.min((long)length, this.checksumPosition - this.writtenBytes));
            this.out.writeBytes(b, offset, lengthToCopy);
            this.writtenBytes += (long)lengthToCopy;
            offset += lengthToCopy;
            length -= lengthToCopy;
            if (this.writtenBytes == this.checksumPosition) {
                this.computeChecksum();
            } else {
                assert (length == 0);
                assert (this.writtenBytes < this.checksumPosition);
            }
        }
        if (0 < length) {
            assert (this.writtenBytes >= this.checksumPosition);
            int checksumIndex = Math.toIntExact(Math.min(8L, this.writtenBytes - this.checksumPosition));
            if (checksumIndex < 8) {
                int lengthToCopy = Math.toIntExact(Math.min(length, 8 - checksumIndex));
                this.footerChecksum.put(b, offset, lengthToCopy);
                this.out.writeBytes(b, offset, lengthToCopy);
                this.writtenBytes += (long)lengthToCopy;
                offset += lengthToCopy;
                length -= lengthToCopy;
            }
        }
        if (0 < length) {
            assert (this.writtenBytes >= this.checksumPosition + 8L);
            this.out.writeBytes(b, offset, length);
            this.writtenBytes += (long)length;
        }
    }

    private void computeChecksum() throws IOException {
        assert (this.writtenBytes == this.checksumPosition) : this.writtenBytes + " vs " + this.checksumPosition;
        assert (this.computedChecksum.position() == 0);
        this.computedChecksum.putLong(this.getChecksum());
    }

    private static String checksumString(ByteBuffer byteBuffer) {
        assert (byteBuffer.remaining() == 0);
        byteBuffer.flip();
        assert (byteBuffer.remaining() == 8);
        return Store.digestToString(byteBuffer.getLong());
    }

    @Nullable
    private String getChecksumsIfInvalid() {
        if (this.writtenBytes != this.metadata.length()) {
            return "actual=<invalid length> footer=<invalid length>";
        }
        if (this.metadata.length() < 8L) {
            return "actual=<too short> footer=<too short>";
        }
        assert (this.computedChecksum.remaining() == 0);
        String computedChecksumString = VerifyingIndexOutput.checksumString(this.computedChecksum);
        if (!this.metadata.checksum().equals(computedChecksumString)) {
            return Strings.format("actual=%s footer=<not checked>", computedChecksumString);
        }
        String footerChecksumString = VerifyingIndexOutput.checksumString(this.footerChecksum);
        if (this.metadata.checksum().equals(footerChecksumString)) {
            return null;
        }
        return Strings.format("actual=%s footer=%s", computedChecksumString, footerChecksumString);
    }

    void verify() throws IOException {
        String checksumsIfInvalid = this.getChecksumsIfInvalid();
        if (checksumsIfInvalid != null) {
            throw new CorruptIndexException(Strings.format("verification failed (hardware problem?) : expected=%s %s writtenLength=%d expectedLength=%d (resource=%s)", this.metadata.checksum(), checksumsIfInvalid, this.writtenBytes, this.metadata.length(), this.metadata), "VerifyingIndexOutput(" + this.metadata.name() + ")");
        }
    }
}

