"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initializeLayoutManager = initializeLayoutManager;
var _fastDeepEqual = _interopRequireDefault(require("fast-deep-equal"));
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _uuid = require("uuid");
var _analytics = require("@kbn/analytics");
var _public = require("@kbn/embeddable-plugin/public");
var _i18n = require("@kbn/i18n");
var _presentationPublishing = require("@kbn/presentation-publishing");
var _std = require("@kbn/std");
var _constants = require("../../../common/constants");
var _dashboard_actions_strings = require("../../dashboard_actions/_dashboard_actions_strings");
var _dashboard_app_strings = require("../../dashboard_app/_dashboard_app_strings");
var _get_panel_placement_settings = require("../../panel_placement/get_panel_placement_settings");
var _place_clone_panel_strategy = require("../../panel_placement/place_clone_panel_strategy");
var _place_new_panel_strategies = require("../../panel_placement/place_new_panel_strategies");
var _plugin_constants = require("../../plugin_constants");
var _kibana_services = require("../../services/kibana_services");
var _telemetry_constants = require("../../utils/telemetry_constants");
var _are_layouts_equal = require("./are_layouts_equal");
var _deserialize_layout = require("./deserialize_layout");
var _serialize_layout = require("./serialize_layout");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function initializeLayoutManager(incomingEmbeddables, initialPanels, trackPanel, getReferences) {
  // --------------------------------------------------------------------------------------
  // Set up panel state manager
  // --------------------------------------------------------------------------------------
  const children$ = new _rxjs.BehaviorSubject({});
  const {
    layout: initialLayout,
    childState: initialChildState
  } = (0, _deserialize_layout.deserializeLayout)(initialPanels, getReferences);
  const layout$ = new _rxjs.BehaviorSubject(initialLayout); // layout is the source of truth for which panels are in the dashboard.
  const gridLayout$ = new _rxjs.BehaviorSubject(transformDashboardLayoutToGridLayout(initialLayout, {})); // source of truth for rendering
  const panelResizeSettings$ = layout$.pipe((0, _rxjs.map)(({
    panels
  }) => {
    return [...new Set(Object.values(panels).map(panel => panel.type))];
  }), (0, _rxjs.distinctUntilChanged)(_fastDeepEqual.default), (0, _rxjs.mergeMap)(async panelTypes => {
    const settingsByPanelType = {};
    await (0, _std.asyncForEach)(panelTypes, async type => {
      const panelSettings = await (0, _get_panel_placement_settings.getPanelSettings)(type);
      if (panelSettings !== null && panelSettings !== void 0 && panelSettings.resizeSettings) settingsByPanelType[type] = panelSettings.resizeSettings;
    });
    return settingsByPanelType;
  }), (0, _rxjs.startWith)({}) // do not block rendering by waiting for these settings
  );

  /** Keep gridLayout$ in sync with layout$ + panelResizeSettings$ */
  const gridLayoutSubscription = (0, _rxjs.combineLatest)([layout$, panelResizeSettings$]).subscribe(([layout, panelResizeSettings]) => {
    gridLayout$.next(transformDashboardLayoutToGridLayout(layout, panelResizeSettings));
  });
  let currentChildState = initialChildState; // childState is the source of truth for the state of each panel.

  let lastSavedLayout = initialLayout;
  let lastSavedChildState = initialChildState;
  const resetLayout = () => {
    layout$.next({
      ...lastSavedLayout
    });
    currentChildState = {
      ...lastSavedChildState
    };
    let childrenModified = false;
    const currentChildren = {
      ...children$.value
    };
    for (const uuid of Object.keys(currentChildren)) {
      if (lastSavedLayout.panels[uuid]) {
        const child = currentChildren[uuid];
        if ((0, _presentationPublishing.apiPublishesUnsavedChanges)(child)) child.resetUnsavedChanges();
      } else {
        // if reset resulted in panel removal, we need to update the list of children
        delete currentChildren[uuid];
        delete currentChildState[uuid];
        childrenModified = true;
      }
    }
    if (childrenModified) children$.next(currentChildren);
  };

  // --------------------------------------------------------------------------------------
  // Panel placement functions
  // --------------------------------------------------------------------------------------
  const placeNewPanel = async (uuid, panelPackage, grid) => {
    const {
      panelType: type,
      serializedState
    } = panelPackage;
    if (grid) {
      return {
        ...layout$.value,
        panels: {
          ...layout$.value.panels,
          [uuid]: {
            grid,
            type
          }
        }
      };
    }
    const panelSettings = await (0, _get_panel_placement_settings.getPanelSettings)(type, serializedState);
    const panelPlacementSettings = {
      strategy: _plugin_constants.PanelPlacementStrategy.findTopLeftMostOpenSpace,
      height: _constants.DEFAULT_PANEL_HEIGHT,
      width: _constants.DEFAULT_PANEL_WIDTH,
      ...(panelSettings === null || panelSettings === void 0 ? void 0 : panelSettings.placementSettings)
    };
    const {
      newPanelPlacement,
      otherPanels
    } = (0, _place_new_panel_strategies.runPanelPlacementStrategy)(panelPlacementSettings === null || panelPlacementSettings === void 0 ? void 0 : panelPlacementSettings.strategy, {
      currentPanels: layout$.value.panels,
      height: panelPlacementSettings.height,
      width: panelPlacementSettings.width
    });
    return {
      ...layout$.value,
      panels: {
        ...otherPanels,
        [uuid]: {
          grid: newPanelPlacement,
          type
        }
      }
    };
  };

  // --------------------------------------------------------------------------------------
  // Place the incoming embeddables if there is at least one
  // --------------------------------------------------------------------------------------
  if (incomingEmbeddables !== null && incomingEmbeddables !== void 0 && incomingEmbeddables.length) {
    for (const incomingEmbeddable of incomingEmbeddables) {
      var _incomingEmbeddable$e, _size$width, _size$height;
      const {
        serializedState,
        size,
        type
      } = incomingEmbeddable;
      const uuid = (_incomingEmbeddable$e = incomingEmbeddable.embeddableId) !== null && _incomingEmbeddable$e !== void 0 ? _incomingEmbeddable$e : (0, _uuid.v4)();
      const existingPanel = layout$.value.panels[uuid];
      const sameType = (existingPanel === null || existingPanel === void 0 ? void 0 : existingPanel.type) === type;
      const grid = existingPanel ? existingPanel.grid : (0, _place_new_panel_strategies.runPanelPlacementStrategy)(_plugin_constants.PanelPlacementStrategy.findTopLeftMostOpenSpace, {
        width: (_size$width = size === null || size === void 0 ? void 0 : size.width) !== null && _size$width !== void 0 ? _size$width : _constants.DEFAULT_PANEL_WIDTH,
        height: (_size$height = size === null || size === void 0 ? void 0 : size.height) !== null && _size$height !== void 0 ? _size$height : _constants.DEFAULT_PANEL_HEIGHT,
        currentPanels: layout$.value.panels
      }).newPanelPlacement;
      currentChildState[uuid] = {
        rawState: {
          ...(sameType && currentChildState[uuid] ? currentChildState[uuid].rawState : {}),
          ...serializedState.rawState
        },
        references: serializedState === null || serializedState === void 0 ? void 0 : serializedState.references
      };
      layout$.next({
        ...layout$.value,
        panels: {
          ...layout$.value.panels,
          [uuid]: {
            grid,
            type
          }
        }
      });
      trackPanel.setScrollToPanelId(uuid);
      trackPanel.setHighlightPanelId(uuid);
    }
  }
  function getDashboardPanelFromId(panelId) {
    const childLayout = layout$.value.panels[panelId];
    const childApi = children$.value[panelId];
    if (!childApi || !childLayout) throw new _public.PanelNotFoundError();
    return {
      type: childLayout.type,
      grid: childLayout.grid,
      serializedState: (0, _presentationPublishing.apiHasSerializableState)(childApi) ? childApi.serializeState() : {
        rawState: {}
      }
    };
  }
  async function getPanelTitles() {
    const titles = [];
    await (0, _std.asyncForEach)(Object.keys(layout$.value.panels), async id => {
      const childApi = await getChildApi(id);
      const title = (0, _presentationPublishing.apiPublishesTitle)(childApi) ? (0, _presentationPublishing.getTitle)(childApi) : '';
      if (title) titles.push(title);
    });
    return titles;
  }

  // --------------------------------------------------------------------------------------
  // API definition
  // --------------------------------------------------------------------------------------
  const addNewPanel = async (panelPackage, displaySuccessMessage, grid) => {
    const {
      panelType: type,
      serializedState,
      maybePanelId
    } = panelPackage;
    const uuid = maybePanelId !== null && maybePanelId !== void 0 ? maybePanelId : (0, _uuid.v4)();
    _kibana_services.usageCollectionService === null || _kibana_services.usageCollectionService === void 0 ? void 0 : _kibana_services.usageCollectionService.reportUiCounter(_telemetry_constants.DASHBOARD_UI_METRIC_ID, _analytics.METRIC_TYPE.CLICK, type);
    if (serializedState) currentChildState[uuid] = serializedState;
    layout$.next(await placeNewPanel(uuid, panelPackage, grid));
    if (displaySuccessMessage) {
      var _serializedState$rawS;
      const title = serializedState === null || serializedState === void 0 ? void 0 : (_serializedState$rawS = serializedState.rawState) === null || _serializedState$rawS === void 0 ? void 0 : _serializedState$rawS.title;
      _kibana_services.coreServices.notifications.toasts.addSuccess({
        title: (0, _dashboard_app_strings.getPanelAddedSuccessString)(title),
        'data-test-subj': 'addEmbeddableToDashboardSuccess'
      });
    }
    trackPanel.setScrollToPanelId(uuid);
    trackPanel.setHighlightPanelId(uuid);
    return await getChildApi(uuid);
  };
  const removePanel = uuid => {
    const panels = {
      ...layout$.value.panels
    };
    if (panels[uuid]) {
      delete panels[uuid];
      layout$.next({
        ...layout$.value,
        panels
      });
    }
    const children = {
      ...children$.value
    };
    if (children[uuid]) {
      delete children[uuid];
      children$.next(children);
    }
    if (currentChildState[uuid]) {
      delete currentChildState[uuid];
    }
  };
  const replacePanel = async (idToRemove, panelPackage) => {
    var _layout$$value$panels;
    const existingGridData = (_layout$$value$panels = layout$.value.panels[idToRemove]) === null || _layout$$value$panels === void 0 ? void 0 : _layout$$value$panels.grid;
    if (!existingGridData) throw new _public.PanelNotFoundError();
    removePanel(idToRemove);
    const newPanel = await addNewPanel(panelPackage, false, existingGridData);
    return newPanel.uuid;
  };
  const duplicatePanel = async uuidToDuplicate => {
    var _getTitle;
    const layoutItemToDuplicate = layout$.value.panels[uuidToDuplicate];
    const apiToDuplicate = children$.value[uuidToDuplicate];
    if (!apiToDuplicate || !layoutItemToDuplicate) throw new _public.PanelNotFoundError();
    const allTitles = await getPanelTitles();
    const lastTitle = (0, _presentationPublishing.apiPublishesTitle)(apiToDuplicate) ? (_getTitle = (0, _presentationPublishing.getTitle)(apiToDuplicate)) !== null && _getTitle !== void 0 ? _getTitle : '' : '';
    const newTitle = getClonedPanelTitle(allTitles, lastTitle);
    const uuidOfDuplicate = (0, _uuid.v4)();
    const serializedState = (0, _presentationPublishing.apiHasLibraryTransforms)(apiToDuplicate) ? apiToDuplicate.getSerializedStateByValue() : apiToDuplicate.serializeState();
    serializedState.rawState.title = newTitle;
    currentChildState[uuidOfDuplicate] = serializedState;
    const {
      newPanelPlacement,
      otherPanels
    } = (0, _place_clone_panel_strategy.placeClonePanel)({
      width: layoutItemToDuplicate.grid.w,
      height: layoutItemToDuplicate.grid.h,
      sectionId: layoutItemToDuplicate.grid.sectionId,
      currentPanels: layout$.value.panels,
      placeBesideId: uuidToDuplicate
    });
    layout$.next({
      ...layout$.value,
      panels: {
        ...otherPanels,
        [uuidOfDuplicate]: {
          grid: {
            ...newPanelPlacement,
            sectionId: layoutItemToDuplicate.grid.sectionId
          },
          type: layoutItemToDuplicate.type
        }
      }
    });
    _kibana_services.coreServices.notifications.toasts.addSuccess({
      title: _dashboard_actions_strings.dashboardClonePanelActionStrings.getSuccessMessage(),
      'data-test-subj': 'addObjectToContainerSuccess'
    });
    trackPanel.setScrollToPanelId(uuidOfDuplicate);
    trackPanel.setHighlightPanelId(uuidOfDuplicate);
  };
  const getChildApi = async uuid => {
    const panelLayout = layout$.value.panels[uuid];
    if (!panelLayout) throw new _public.PanelNotFoundError();
    if (children$.value[uuid]) return children$.value[uuid];

    // if the panel is in a collapsed section and has never been built, then childApi will be undefined
    if (isSectionCollapsed(panelLayout.grid.sectionId)) {
      return undefined;
    }
    return new Promise(resolve => {
      const subscription = (0, _rxjs.merge)(children$, layout$).subscribe(() => {
        if (children$.value[uuid]) {
          subscription.unsubscribe();
          resolve(children$.value[uuid]);
        }

        // If we hit this, the panel was removed before the embeddable finished loading.
        if (layout$.value.panels[uuid] === undefined) {
          subscription.unsubscribe();
          resolve(undefined);
        }
      });
    });
  };
  function isSectionCollapsed(sectionId) {
    const {
      sections
    } = layout$.getValue();
    return Boolean(sectionId && sections[sectionId].collapsed);
  }
  return {
    internalApi: {
      getSerializedStateForPanel: panelId => currentChildState[panelId],
      getLastSavedStateForPanel: panelId => lastSavedChildState[panelId],
      layout$,
      gridLayout$,
      reset: resetLayout,
      serializeLayout: () => (0, _serialize_layout.serializeLayout)(layout$.value, currentChildState),
      startComparing$: lastSavedState$ => {
        return layout$.pipe((0, _rxjs.debounceTime)(100), (0, _rxjs.combineLatestWith)(lastSavedState$.pipe((0, _rxjs.map)(lastSaved => (0, _deserialize_layout.deserializeLayout)(lastSaved.panels, getReferences)), (0, _rxjs.tap)(({
          layout,
          childState
        }) => {
          lastSavedChildState = childState;
          lastSavedLayout = layout;
        }))), (0, _rxjs.map)(([currentLayout]) => {
          if (!(0, _are_layouts_equal.areLayoutsEqual)(lastSavedLayout, currentLayout)) {
            (0, _presentationPublishing.logStateDiff)('dashboard layout', lastSavedLayout, currentLayout);
            return {
              panels: (0, _serialize_layout.serializeLayout)(currentLayout, currentChildState).panels
            };
          }
          return {};
        }));
      },
      registerChildApi: api => {
        children$.next({
          ...children$.value,
          [api.uuid]: api
        });
      },
      setChildState: (uuid, state) => {
        currentChildState[uuid] = state;
      },
      isSectionCollapsed
    },
    api: {
      /** Panels */
      children$,
      getChildApi,
      addNewPanel,
      removePanel,
      replacePanel,
      duplicatePanel,
      getDashboardPanelFromId,
      getPanelCount: () => Object.keys(layout$.value.panels).length,
      canRemovePanels: () => trackPanel.expandedPanelId$.value === undefined,
      /** Sections */
      addNewSection: () => {
        const currentLayout = layout$.getValue();

        // find the max y so we know where to add the section
        let maxY = 0;
        [...Object.values(currentLayout.panels), ...Object.values(currentLayout.sections)].forEach(widget => {
          const {
            y,
            h
          } = {
            h: 1,
            ...widget.grid
          };
          maxY = Math.max(maxY, y + h);
        });

        // add the new section
        const sections = {
          ...currentLayout.sections
        };
        const newId = (0, _uuid.v4)();
        sections[newId] = {
          grid: {
            y: maxY
          },
          title: _i18n.i18n.translate('dashboard.defaultSectionTitle', {
            defaultMessage: 'New collapsible section'
          }),
          collapsed: false
        };
        layout$.next({
          ...currentLayout,
          sections
        });
        trackPanel.scrollToBottom$.next();
      }
    },
    cleanup: () => {
      gridLayoutSubscription.unsubscribe();
    }
  };
}
function getClonedPanelTitle(panelTitles, rawTitle) {
  if (rawTitle === '') return '';
  const clonedTag = _dashboard_actions_strings.dashboardClonePanelActionStrings.getClonedTag();
  const cloneRegex = new RegExp(`\\(${clonedTag}\\)`, 'g');
  const cloneNumberRegex = new RegExp(`\\(${clonedTag} [0-9]+\\)`, 'g');
  const baseTitle = rawTitle.replace(cloneNumberRegex, '').replace(cloneRegex, '').trim();
  const similarTitles = (0, _lodash.filter)(panelTitles, title => {
    return title.startsWith(baseTitle);
  });
  const cloneNumbers = (0, _lodash.map)(similarTitles, title => {
    if (title.match(cloneRegex)) return 0;
    const cloneTag = title.match(cloneNumberRegex);
    return cloneTag ? parseInt(cloneTag[0].replace(/[^0-9.]/g, ''), 10) : -1;
  });
  const similarBaseTitlesCount = (0, _lodash.max)(cloneNumbers) || 0;
  return similarBaseTitlesCount < 0 ? baseTitle + ` (${clonedTag})` : baseTitle + ` (${clonedTag} ${similarBaseTitlesCount + 1})`;
}
const transformDashboardLayoutToGridLayout = (layout, resizeSettings) => {
  const newLayout = {};
  Object.keys(layout.sections).forEach(sectionId => {
    const section = layout.sections[sectionId];
    newLayout[sectionId] = {
      id: sectionId,
      type: 'section',
      row: section.grid.y,
      isCollapsed: Boolean(section.collapsed),
      title: section.title,
      panels: {}
    };
  });
  Object.keys(layout.panels).forEach(panelId => {
    const grid = layout.panels[panelId].grid;
    const type = layout.panels[panelId].type;
    const basePanel = {
      id: panelId,
      row: grid.y,
      column: grid.x,
      width: grid.w,
      height: grid.h,
      resizeOptions: resizeSettings[type]
    };
    if (grid.sectionId) {
      newLayout[grid.sectionId].panels[panelId] = basePanel;
    } else {
      newLayout[panelId] = {
        ...basePanel,
        type: 'panel'
      };
    }
  });
  return newLayout;
};