/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.transport;

import java.io.Closeable;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Executor;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.function.Supplier;
import org.apache.lucene.util.SetOnce;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.LegacyActionRequest;
import org.elasticsearch.action.admin.cluster.remote.RemoteClusterNodesAction;
import org.elasticsearch.action.admin.cluster.state.ClusterStateResponse;
import org.elasticsearch.action.admin.cluster.state.RemoteClusterStateRequest;
import org.elasticsearch.cluster.ClusterName;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodeRole;
import org.elasticsearch.cluster.node.VersionInformation;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.common.transport.TransportAddress;
import org.elasticsearch.common.util.concurrent.ListenableFuture;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.index.IndexVersion;
import org.elasticsearch.index.IndexVersions;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.ConnectTransportException;
import org.elasticsearch.transport.ConnectionManager;
import org.elasticsearch.transport.ConnectionProfile;
import org.elasticsearch.transport.LinkedProjectConfig;
import org.elasticsearch.transport.NoSeedNodeLeftException;
import org.elasticsearch.transport.RemoteConnectionInfo;
import org.elasticsearch.transport.RemoteConnectionManager;
import org.elasticsearch.transport.RemoteConnectionStrategy;
import org.elasticsearch.transport.Transport;
import org.elasticsearch.transport.TransportException;
import org.elasticsearch.transport.TransportRequest;
import org.elasticsearch.transport.TransportRequestOptions;
import org.elasticsearch.transport.TransportResponse;
import org.elasticsearch.transport.TransportResponseHandler;
import org.elasticsearch.transport.TransportService;
import org.elasticsearch.xcontent.ToXContent;
import org.elasticsearch.xcontent.XContentBuilder;

public class SniffConnectionStrategy
extends RemoteConnectionStrategy {
    static final int CHANNELS_PER_CONNECTION = 6;
    private static final TimeValue SNIFF_REQUEST_TIMEOUT = TimeValue.THIRTY_SECONDS;
    private final List<String> configuredSeedNodes;
    private final List<Supplier<DiscoveryNode>> seedNodes;
    private final int maxNumRemoteConnections;
    private final Predicate<DiscoveryNode> nodePredicate;
    private final SetOnce<ClusterName> remoteClusterName = new SetOnce();
    private final String proxyAddress;
    private final Executor managementExecutor;

    SniffConnectionStrategy(LinkedProjectConfig.SniffLinkedProjectConfig config, TransportService transportService, RemoteConnectionManager connectionManager) {
        this(config, config.seedNodes().stream().map(seedAddress -> () -> SniffConnectionStrategy.resolveSeedNode(config.linkedProjectAlias(), seedAddress, config.proxyAddress())).toList(), transportService, connectionManager);
    }

    SniffConnectionStrategy(LinkedProjectConfig.SniffLinkedProjectConfig config, List<Supplier<DiscoveryNode>> seedNodesSupplier, TransportService transportService, RemoteConnectionManager connectionManager) {
        super(config, transportService, connectionManager);
        this.proxyAddress = config.proxyAddress();
        this.maxNumRemoteConnections = config.maxNumConnections();
        this.nodePredicate = config.nodePredicate();
        this.configuredSeedNodes = config.seedNodes();
        this.seedNodes = seedNodesSupplier;
        this.managementExecutor = transportService.getThreadPool().executor("management");
    }

    static Writeable.Reader<RemoteConnectionInfo.ModeInfo> infoReader() {
        return SniffModeInfo::new;
    }

    @Override
    protected boolean shouldOpenMoreConnections() {
        return this.connectionManager.size() < this.maxNumRemoteConnections;
    }

    @Override
    protected boolean strategyMustBeRebuilt(LinkedProjectConfig config) {
        assert (config instanceof LinkedProjectConfig.SniffLinkedProjectConfig) : "expected config to be of type " + String.valueOf(LinkedProjectConfig.SniffLinkedProjectConfig.class);
        LinkedProjectConfig.SniffLinkedProjectConfig sniffConfig = (LinkedProjectConfig.SniffLinkedProjectConfig)config;
        return sniffConfig.maxNumConnections() != this.maxNumRemoteConnections || SniffConnectionStrategy.seedsChanged(this.configuredSeedNodes, sniffConfig.seedNodes()) || SniffConnectionStrategy.proxyChanged(this.proxyAddress, sniffConfig.proxyAddress());
    }

    @Override
    protected RemoteConnectionStrategy.ConnectionStrategy strategyType() {
        return RemoteConnectionStrategy.ConnectionStrategy.SNIFF;
    }

    @Override
    protected void connectImpl(ActionListener<Void> listener) {
        this.collectRemoteNodes(this.seedNodes.iterator(), listener);
    }

    @Override
    protected RemoteConnectionInfo.ModeInfo getModeInfo() {
        return new SniffModeInfo(this.configuredSeedNodes, this.maxNumRemoteConnections, this.connectionManager.size());
    }

    private void collectRemoteNodes(Iterator<Supplier<DiscoveryNode>> seedNodesSuppliers, ActionListener<Void> listener) {
        if (Thread.currentThread().isInterrupted()) {
            listener.onFailure(new InterruptedException("remote connect thread got interrupted"));
            return;
        }
        if (seedNodesSuppliers.hasNext()) {
            Consumer<Exception> onFailure = e -> {
                if (SniffConnectionStrategy.isRetryableException(e) && seedNodesSuppliers.hasNext()) {
                    this.logger.debug(() -> "fetching nodes from external cluster [" + this.clusterAlias + "] failed moving to next seed node", (Throwable)e);
                    this.collectRemoteNodes(seedNodesSuppliers, listener);
                } else {
                    listener.onFailure((Exception)e);
                }
            };
            DiscoveryNode seedNode = seedNodesSuppliers.next().get();
            this.logger.trace("[{}] opening transient connection to seed node: [{}]", (Object)this.clusterAlias, (Object)seedNode);
            ListenableFuture<Transport.Connection> openConnectionStep = new ListenableFuture<Transport.Connection>();
            try {
                this.connectionManager.openConnection(seedNode, null, openConnectionStep);
            }
            catch (Exception e2) {
                onFailure.accept(e2);
            }
            ListenableFuture handshakeStep = new ListenableFuture();
            openConnectionStep.addListener(ActionListener.wrap(connection -> {
                ConnectionProfile connectionProfile = this.connectionManager.getConnectionProfile();
                this.transportService.handshake((Transport.Connection)connection, connectionProfile.getHandshakeTimeout(), this.getRemoteClusterNamePredicate(), handshakeStep);
            }, onFailure));
            ListenableFuture fullConnectionStep = new ListenableFuture();
            handshakeStep.addListener(ActionListener.wrap(handshakeResponse -> {
                DiscoveryNode handshakeNode = handshakeResponse.getDiscoveryNode();
                if (this.nodePredicate.test(handshakeNode) && this.shouldOpenMoreConnections()) {
                    this.logger.trace("[{}] opening managed connection to seed node: [{}] proxy address: [{}]", (Object)this.clusterAlias, (Object)handshakeNode, (Object)this.proxyAddress);
                    DiscoveryNode handshakeNodeWithProxy = SniffConnectionStrategy.maybeAddProxyAddress(this.proxyAddress, handshakeNode);
                    this.connectionManager.connectToRemoteClusterNode(handshakeNodeWithProxy, this.getConnectionValidator(handshakeNodeWithProxy), fullConnectionStep);
                } else {
                    fullConnectionStep.onResponse(null);
                }
            }, e -> {
                Transport.Connection connection = (Transport.Connection)openConnectionStep.result();
                DiscoveryNode node = connection.getNode();
                this.logger.debug(() -> Strings.format((String)"[%s] failed to handshake with seed node: [%s]", (Object[])new Object[]{this.clusterAlias, node}), (Throwable)e);
                IOUtils.closeWhileHandlingException((Closeable)connection);
                onFailure.accept((Exception)e);
            }));
            fullConnectionStep.addListener(ActionListener.wrap(aVoid -> {
                AbstractSniffResponseHandler sniffResponseHandler;
                LegacyActionRequest request;
                String action;
                if (this.remoteClusterName.get() == null) {
                    TransportService.HandshakeResponse handshakeResponse = (TransportService.HandshakeResponse)handshakeStep.result();
                    assert (handshakeResponse.getClusterName().value() != null);
                    this.remoteClusterName.set((Object)handshakeResponse.getClusterName());
                }
                Transport.Connection connection = (Transport.Connection)openConnectionStep.result();
                ThreadPool threadPool = this.transportService.getThreadPool();
                ThreadContext threadContext = threadPool.getThreadContext();
                if ("_remote_cluster".equals(this.connectionManager.getConnectionProfile().getTransportProfile())) {
                    action = RemoteClusterNodesAction.TYPE.name();
                    request = RemoteClusterNodesAction.Request.REMOTE_CLUSTER_SERVER_NODES;
                    sniffResponseHandler = new RemoteClusterNodesSniffResponseHandler(this, connection, listener, seedNodesSuppliers);
                } else {
                    action = "cluster:monitor/state";
                    RemoteClusterStateRequest clusterStateRequest = new RemoteClusterStateRequest(SNIFF_REQUEST_TIMEOUT);
                    clusterStateRequest.clear();
                    clusterStateRequest.nodes(true);
                    request = clusterStateRequest;
                    sniffResponseHandler = new ClusterStateSniffResponseHandler(this, connection, listener, seedNodesSuppliers);
                }
                try (ThreadContext.StoredContext ignored = threadContext.newEmptySystemContext();){
                    this.transportService.sendRequest(connection, action, (TransportRequest)request, TransportRequestOptions.EMPTY, new TransportService.ContextRestoreResponseHandler<RemoteClusterNodesAction.Response>(threadContext.newRestorableContext(false), sniffResponseHandler));
                }
            }, e -> {
                Transport.Connection connection = (Transport.Connection)openConnectionStep.result();
                DiscoveryNode node = connection.getNode();
                this.logger.debug(() -> Strings.format((String)"[%s] failed to open managed connection to seed node: [%s]", (Object[])new Object[]{this.clusterAlias, node}), (Throwable)e);
                IOUtils.closeWhileHandlingException((Closeable)connection);
                onFailure.accept((Exception)e);
            }));
        } else {
            listener.onFailure(new NoSeedNodeLeftException("no seed node left for cluster: [" + this.clusterAlias + "]"));
        }
    }

    private ConnectionManager.ConnectionValidator getConnectionValidator(DiscoveryNode node) {
        return (connection, profile, listener) -> {
            assert (profile.getTransportProfile().equals(this.connectionManager.getConnectionProfile().getTransportProfile())) : "transport profile must be consistent between the connection manager and the actual profile";
            this.transportService.connectionValidator(node).validate(RemoteConnectionManager.wrapConnectionWithRemoteClusterInfo(connection, this.clusterAlias, this.connectionManager.getCredentialsManager()), profile, listener);
        };
    }

    private Predicate<ClusterName> getRemoteClusterNamePredicate() {
        return new Predicate<ClusterName>(){

            @Override
            public boolean test(ClusterName c) {
                return SniffConnectionStrategy.this.remoteClusterName.get() == null || c.equals(SniffConnectionStrategy.this.remoteClusterName.get());
            }

            public String toString() {
                return SniffConnectionStrategy.this.remoteClusterName.get() == null ? "any cluster name" : "expected remote cluster name [" + ((ClusterName)SniffConnectionStrategy.this.remoteClusterName.get()).value() + "]";
            }
        };
    }

    private static DiscoveryNode resolveSeedNode(String clusterAlias, String address, String proxyAddress) {
        VersionInformation seedVersion = new VersionInformation(Version.CURRENT.minimumCompatibilityVersion(), IndexVersions.MINIMUM_COMPATIBLE, IndexVersions.MINIMUM_READONLY_COMPATIBLE, IndexVersion.current());
        if (proxyAddress == null || proxyAddress.isEmpty()) {
            TransportAddress transportAddress = new TransportAddress(SniffConnectionStrategy.parseConfiguredAddress(address));
            return new DiscoveryNode(null, clusterAlias + "#" + String.valueOf(transportAddress), transportAddress, Collections.emptyMap(), DiscoveryNodeRole.roles(), seedVersion);
        }
        TransportAddress transportAddress = new TransportAddress(SniffConnectionStrategy.parseConfiguredAddress(proxyAddress));
        String hostName = RemoteConnectionStrategy.parseHost(proxyAddress);
        return new DiscoveryNode(null, clusterAlias + "#" + address, UUIDs.randomBase64UUID(), hostName, address, transportAddress, Collections.singletonMap("server_name", hostName), DiscoveryNodeRole.roles(), seedVersion);
    }

    private static DiscoveryNode maybeAddProxyAddress(String proxyAddress, DiscoveryNode node) {
        if (proxyAddress == null || proxyAddress.isEmpty()) {
            return node;
        }
        InetSocketAddress proxyInetAddress = SniffConnectionStrategy.parseConfiguredAddress(proxyAddress);
        return new DiscoveryNode(node.getName(), node.getId(), node.getEphemeralId(), node.getHostName(), node.getHostAddress(), new TransportAddress(proxyInetAddress), node.getAttributes(), node.getRoles(), node.getVersionInformation());
    }

    private static boolean seedsChanged(List<String> oldSeedNodes, List<String> newSeedNodes) {
        if (oldSeedNodes.size() != newSeedNodes.size()) {
            return true;
        }
        HashSet<String> oldSeeds = new HashSet<String>(oldSeedNodes);
        HashSet<String> newSeeds = new HashSet<String>(newSeedNodes);
        return !oldSeeds.equals(newSeeds);
    }

    private static boolean proxyChanged(String oldProxy, String newProxy) {
        if (oldProxy == null || oldProxy.isEmpty()) {
            return !(newProxy == null || newProxy.isEmpty());
        }
        return !Objects.equals(oldProxy, newProxy);
    }

    public static class SniffModeInfo
    implements RemoteConnectionInfo.ModeInfo {
        final List<String> seedNodes;
        final int maxConnectionsPerCluster;
        final int numNodesConnected;

        public SniffModeInfo(List<String> seedNodes, int maxConnectionsPerCluster, int numNodesConnected) {
            this.seedNodes = seedNodes;
            this.maxConnectionsPerCluster = maxConnectionsPerCluster;
            this.numNodesConnected = numNodesConnected;
        }

        private SniffModeInfo(StreamInput input) throws IOException {
            this.seedNodes = Arrays.asList(input.readStringArray());
            this.maxConnectionsPerCluster = input.readVInt();
            this.numNodesConnected = input.readVInt();
        }

        public XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
            builder.startArray("seeds");
            for (String address : this.seedNodes) {
                builder.value(address);
            }
            builder.endArray();
            builder.field("num_nodes_connected", this.numNodesConnected);
            builder.field("max_connections_per_cluster", this.maxConnectionsPerCluster);
            return builder;
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            out.writeStringCollection(this.seedNodes);
            out.writeVInt(this.maxConnectionsPerCluster);
            out.writeVInt(this.numNodesConnected);
        }

        @Override
        public boolean isConnected() {
            return this.numNodesConnected > 0;
        }

        @Override
        public String modeName() {
            return "sniff";
        }

        @Override
        public RemoteConnectionStrategy.ConnectionStrategy modeType() {
            return RemoteConnectionStrategy.ConnectionStrategy.SNIFF;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            SniffModeInfo sniff = (SniffModeInfo)o;
            return this.maxConnectionsPerCluster == sniff.maxConnectionsPerCluster && this.numNodesConnected == sniff.numNodesConnected && Objects.equals(this.seedNodes, sniff.seedNodes);
        }

        public int hashCode() {
            return Objects.hash(this.seedNodes, this.maxConnectionsPerCluster, this.numNodesConnected);
        }

        public String toString() {
            return "SniffModeInfo{seedNodes=" + String.valueOf(this.seedNodes) + ", maxConnectionsPerCluster=" + this.maxConnectionsPerCluster + ", numNodesConnected=" + this.numNodesConnected + "}";
        }
    }

    private class RemoteClusterNodesSniffResponseHandler
    extends AbstractSniffResponseHandler<RemoteClusterNodesAction.Response> {
        RemoteClusterNodesSniffResponseHandler(SniffConnectionStrategy sniffConnectionStrategy, Transport.Connection connection, ActionListener<Void> listener, Iterator<Supplier<DiscoveryNode>> seedNodes) {
            super(connection, listener, seedNodes);
        }

        @Override
        public RemoteClusterNodesAction.Response read(StreamInput in) throws IOException {
            return new RemoteClusterNodesAction.Response(in);
        }

        @Override
        public void handleResponse(RemoteClusterNodesAction.Response response) {
            this.handleNodes(response.getNodes().iterator());
        }
    }

    private class ClusterStateSniffResponseHandler
    extends AbstractSniffResponseHandler<ClusterStateResponse> {
        ClusterStateSniffResponseHandler(SniffConnectionStrategy sniffConnectionStrategy, Transport.Connection connection, ActionListener<Void> listener, Iterator<Supplier<DiscoveryNode>> seedNodes) {
            super(connection, listener, seedNodes);
        }

        @Override
        public ClusterStateResponse read(StreamInput in) throws IOException {
            return new ClusterStateResponse(in);
        }

        @Override
        public void handleResponse(ClusterStateResponse response) {
            this.handleNodes(response.getState().nodes().getNodes().values().iterator());
        }
    }

    private abstract class AbstractSniffResponseHandler<T extends TransportResponse>
    implements TransportResponseHandler<T> {
        private final Transport.Connection connection;
        private final ActionListener<Void> listener;
        private final Iterator<Supplier<DiscoveryNode>> seedNodes;

        AbstractSniffResponseHandler(Transport.Connection connection, ActionListener<Void> listener, Iterator<Supplier<DiscoveryNode>> seedNodes) {
            this.connection = connection;
            this.listener = listener;
            this.seedNodes = seedNodes;
        }

        protected void handleNodes(final Iterator<DiscoveryNode> nodesIter) {
            while (nodesIter.hasNext()) {
                final DiscoveryNode node = nodesIter.next();
                if (!SniffConnectionStrategy.this.nodePredicate.test(node) || !SniffConnectionStrategy.this.shouldOpenMoreConnections()) continue;
                SniffConnectionStrategy.this.logger.trace("[{}] opening managed connection to node: [{}] proxy address: [{}]", (Object)SniffConnectionStrategy.this.clusterAlias, (Object)node, (Object)SniffConnectionStrategy.this.proxyAddress);
                DiscoveryNode nodeWithProxy = SniffConnectionStrategy.maybeAddProxyAddress(SniffConnectionStrategy.this.proxyAddress, node);
                SniffConnectionStrategy.this.connectionManager.connectToRemoteClusterNode(nodeWithProxy, SniffConnectionStrategy.this.getConnectionValidator(node), new ActionListener<Void>(){

                    @Override
                    public void onResponse(Void aVoid) {
                        AbstractSniffResponseHandler.this.handleNodes(nodesIter);
                    }

                    @Override
                    public void onFailure(Exception e) {
                        if (e instanceof ConnectTransportException || e instanceof IllegalStateException) {
                            SniffConnectionStrategy.this.logger.debug(() -> Strings.format((String)"[%s] failed to open managed connection to node [%s]", (Object[])new Object[]{SniffConnectionStrategy.this.clusterAlias, node}), (Throwable)e);
                            AbstractSniffResponseHandler.this.handleNodes(nodesIter);
                        } else {
                            SniffConnectionStrategy.this.logger.warn(() -> Strings.format((String)"[%s] failed to open managed connection to node [%s]", (Object[])new Object[]{SniffConnectionStrategy.this.clusterAlias, node}), (Throwable)e);
                            IOUtils.closeWhileHandlingException((Closeable)AbstractSniffResponseHandler.this.connection);
                            SniffConnectionStrategy.this.collectRemoteNodes(AbstractSniffResponseHandler.this.seedNodes, AbstractSniffResponseHandler.this.listener);
                        }
                    }
                });
                return;
            }
            IOUtils.closeWhileHandlingException((Closeable)this.connection);
            int openConnections = SniffConnectionStrategy.this.connectionManager.size();
            if (openConnections == 0) {
                this.listener.onFailure(new IllegalStateException("Unable to open any connections to remote cluster [" + SniffConnectionStrategy.this.clusterAlias + "]"));
            } else {
                this.listener.onResponse(null);
            }
        }

        @Override
        public void handleException(TransportException exp) {
            SniffConnectionStrategy.this.logger.warn(() -> "fetching nodes from external cluster " + SniffConnectionStrategy.this.clusterAlias + " failed", (Throwable)exp);
            try {
                IOUtils.closeWhileHandlingException((Closeable)this.connection);
            }
            finally {
                SniffConnectionStrategy.this.collectRemoteNodes(this.seedNodes, this.listener);
            }
        }

        @Override
        public Executor executor() {
            return SniffConnectionStrategy.this.managementExecutor;
        }
    }
}

