/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.node;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.support.PlainActionFuture;
import org.elasticsearch.action.support.RefCountingListener;
import org.elasticsearch.action.support.SubscribableListener;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.http.HttpServerTransport;
import org.elasticsearch.node.internal.TerminationHandler;
import org.elasticsearch.tasks.TaskManager;

public class ShutdownPrepareService {
    public static final Setting<TimeValue> MAXIMUM_SHUTDOWN_TIMEOUT_SETTING = Setting.positiveTimeSetting("node.maximum_shutdown_grace_period", TimeValue.ZERO, Setting.Property.NodeScope);
    public static final Setting<TimeValue> MAXIMUM_REINDEXING_TIMEOUT_SETTING = Setting.positiveTimeSetting("node.maximum_reindexing_grace_period", TimeValue.timeValueSeconds(10L), Setting.Property.NodeScope);
    private final Logger logger = LogManager.getLogger(ShutdownPrepareService.class);
    private final TimeValue maxTimeout;
    private final List<ShutdownHook> hooks = new ArrayList<ShutdownHook>();
    private volatile boolean isShuttingDown = false;

    public ShutdownPrepareService(Settings settings, HttpServerTransport httpServerTransport, TaskManager taskManager, TerminationHandler terminationHandler) {
        this.maxTimeout = MAXIMUM_SHUTDOWN_TIMEOUT_SETTING.get(settings);
        TimeValue reindexTimeout = MAXIMUM_REINDEXING_TIMEOUT_SETTING.get(settings);
        this.addShutdownHook("http-server-transport-stop", httpServerTransport::close);
        this.addShutdownHook("async-search-stop", () -> this.awaitSearchTasksComplete(this.maxTimeout, taskManager));
        this.addShutdownHook("reindex-stop", () -> this.awaitReindexTasksComplete(reindexTimeout, taskManager));
        if (terminationHandler != null) {
            this.addShutdownHook("termination-handler-stop", terminationHandler::handleTermination);
        }
    }

    public void addShutdownHook(String name, Runnable action) {
        this.hooks.add(new ShutdownHook(name, action));
    }

    public boolean isShuttingDown() {
        return this.isShuttingDown;
    }

    public void prepareForShutdown() {
        assert (!this.isShuttingDown);
        this.isShuttingDown = true;
        record Stopper(String name, SubscribableListener<Void> listener) {
            boolean isIncomplete() {
                return !this.listener().isDone();
            }
        }
        ArrayList<Stopper> stoppers = new ArrayList<Stopper>();
        PlainActionFuture<Void> allStoppersFuture = new PlainActionFuture<Void>();
        try (RefCountingListener listeners = new RefCountingListener(allStoppersFuture);){
            for (ShutdownHook hook : this.hooks) {
                Stopper stopper = new Stopper(hook.name(), new SubscribableListener<Void>());
                stoppers.add(stopper);
                stopper.listener().addListener(listeners.acquire());
                new Thread(() -> {
                    try {
                        hook.action.run();
                    }
                    catch (Exception ex) {
                        this.logger.warn("unexpected exception in shutdown task [" + stopper.name() + "]", (Throwable)ex);
                    }
                    finally {
                        stopper.listener().onResponse(null);
                    }
                }, stopper.name()).start();
            }
        }
        Supplier<String> incompleteStoppersDescriber = () -> stoppers.stream().filter(Stopper::isIncomplete).map(Stopper::name).collect(Collectors.joining(", ", "[", "]"));
        try {
            if (TimeValue.ZERO.equals(this.maxTimeout)) {
                allStoppersFuture.get();
            } else {
                allStoppersFuture.get(this.maxTimeout.millis(), TimeUnit.MILLISECONDS);
            }
        }
        catch (ExecutionException e) {
            assert (false) : e;
            this.logger.warn("failed during graceful shutdown tasks", (Throwable)e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.logger.warn("interrupted while waiting for graceful shutdown tasks: " + incompleteStoppersDescriber.get(), (Throwable)e);
        }
        catch (TimeoutException e) {
            this.logger.warn("timed out while waiting for graceful shutdown tasks: " + incompleteStoppersDescriber.get());
        }
    }

    private void awaitTasksComplete(TimeValue timeout, String taskName, TaskManager taskManager) {
        long millisWaited = 0L;
        while (true) {
            long tasksRemaining;
            if ((tasksRemaining = taskManager.getTasks().values().stream().filter(task -> taskName.equals(task.getAction())).count()) == 0L) {
                this.logger.debug("all " + taskName + " tasks complete");
                return;
            }
            TimeValue pollPeriod = TimeValue.timeValueMillis(500L);
            if (!TimeValue.ZERO.equals(timeout) && (millisWaited += pollPeriod.millis()) >= timeout.millis()) {
                this.logger.warn(Strings.format("timed out after waiting [%s] for [%d] " + taskName + " tasks to finish", timeout.toString(), tasksRemaining));
                return;
            }
            this.logger.debug(Strings.format("waiting for [%s] " + taskName + " tasks to finish, next poll in [%s]", tasksRemaining, pollPeriod));
            try {
                Thread.sleep(pollPeriod.millis());
            }
            catch (InterruptedException ex) {
                this.logger.warn(Strings.format("interrupted while waiting [%s] for [%d] " + taskName + " tasks to finish", timeout.toString(), tasksRemaining));
                return;
            }
        }
    }

    private void awaitSearchTasksComplete(TimeValue asyncSearchTimeout, TaskManager taskManager) {
        this.awaitTasksComplete(asyncSearchTimeout, "indices:data/read/search", taskManager);
    }

    private void awaitReindexTasksComplete(TimeValue asyncReindexTimeout, TaskManager taskManager) {
        this.awaitTasksComplete(asyncReindexTimeout, "indices:data/write/reindex", taskManager);
    }

    private record ShutdownHook(String name, Runnable action) {
    }
}

