/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing.allocation;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.LongSupplier;
import java.util.function.Supplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.cluster.ClusterInfo;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.NodeUsageStatsForThreadPools;
import org.elasticsearch.cluster.node.DiscoveryNodeRole;
import org.elasticsearch.cluster.routing.RerouteService;
import org.elasticsearch.cluster.routing.allocation.WriteLoadConstraintSettings;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.util.set.Sets;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.telemetry.metric.DoubleHistogram;
import org.elasticsearch.telemetry.metric.LongWithAttributes;
import org.elasticsearch.telemetry.metric.MeterRegistry;

public class WriteLoadConstraintMonitor {
    public static final String HOTSPOT_NODES_COUNT_METRIC_NAME = "es.allocator.allocations.node.write_load_hotspot.current";
    public static final String HOTSPOT_DURATION_METRIC_NAME = "es.allocator.allocations.node.write_load_hotspot.duration.histogram";
    private static final Logger logger = LogManager.getLogger(WriteLoadConstraintMonitor.class);
    private static final int MAX_NODE_IDS_IN_MESSAGE = 3;
    private final WriteLoadConstraintSettings writeLoadConstraintSettings;
    private final Supplier<ClusterState> clusterStateSupplier;
    private final LongSupplier currentTimeMillisSupplier;
    private final RerouteService rerouteService;
    private volatile long lastRerouteTimeMillis = 0L;
    private final Map<String, Long> hotspotNodeStartTimes = new HashMap<String, Long>();
    private long hotspotNodeStartTimesLastTerm = -1L;
    private final AtomicLong hotspotNodesCount = new AtomicLong(-1L);
    private final DoubleHistogram hotspotDurationHistogram;

    protected WriteLoadConstraintMonitor(ClusterSettings clusterSettings, LongSupplier currentTimeMillisSupplier, Supplier<ClusterState> clusterStateSupplier, RerouteService rerouteService) {
        this(clusterSettings, currentTimeMillisSupplier, clusterStateSupplier, rerouteService, MeterRegistry.NOOP);
    }

    public WriteLoadConstraintMonitor(ClusterSettings clusterSettings, LongSupplier currentTimeMillisSupplier, Supplier<ClusterState> clusterStateSupplier, RerouteService rerouteService, MeterRegistry meterRegistry) {
        this.writeLoadConstraintSettings = new WriteLoadConstraintSettings(clusterSettings);
        this.clusterStateSupplier = clusterStateSupplier;
        this.currentTimeMillisSupplier = currentTimeMillisSupplier;
        this.rerouteService = rerouteService;
        meterRegistry.registerLongsGauge(HOTSPOT_NODES_COUNT_METRIC_NAME, "Total number of nodes hotspotting with write loads", "unit", this::getHotspotNodesCount);
        this.hotspotDurationHistogram = meterRegistry.registerDoubleHistogram(HOTSPOT_DURATION_METRIC_NAME, "hotspot duration", "s");
    }

    public void onNewInfo(ClusterInfo clusterInfo) {
        ClusterState state = this.clusterStateSupplier.get();
        if (state.blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK)) {
            logger.trace("skipping monitor as the cluster state is not recovered yet");
            return;
        }
        if (this.writeLoadConstraintSettings.getWriteLoadConstraintEnabled().notFullyEnabled()) {
            logger.trace("skipping monitor because the write load decider is not fully enabled");
            return;
        }
        logger.trace("processing new cluster info");
        int numberOfNodes = clusterInfo.getNodeUsageStatsForThreadPools().size();
        HashSet<String> writeNodeIdsExceedingQueueLatencyThreshold = Sets.newHashSetWithExpectedSize(numberOfNodes);
        boolean haveWriteNodesBelowQueueLatencyThreshold = false;
        int totalIngestNodes = 0;
        for (Map.Entry<String, NodeUsageStatsForThreadPools> entry : clusterInfo.getNodeUsageStatsForThreadPools().entrySet()) {
            String nodeId = entry.getKey();
            NodeUsageStatsForThreadPools usageStats = entry.getValue();
            Set<DiscoveryNodeRole> nodeRoles = state.getNodes().get(nodeId).getRoles();
            if (nodeRoles.contains(DiscoveryNodeRole.SEARCH_ROLE) || nodeRoles.contains(DiscoveryNodeRole.ML_ROLE)) continue;
            ++totalIngestNodes;
            NodeUsageStatsForThreadPools.ThreadPoolUsageStats writeThreadPoolStats = usageStats.threadPoolUsageStatsMap().get("write");
            assert (writeThreadPoolStats != null) : "Write thread pool is not publishing usage stats for node [" + nodeId + "]";
            if (writeThreadPoolStats.maxThreadPoolQueueLatencyMillis() >= this.writeLoadConstraintSettings.getQueueLatencyThreshold().millis()) {
                writeNodeIdsExceedingQueueLatencyThreshold.add(nodeId);
                continue;
            }
            haveWriteNodesBelowQueueLatencyThreshold = true;
        }
        long currentTimeMillis = this.currentTimeMillisSupplier.getAsLong();
        Set<String> lastHotspotNodes = this.recordHotspotDurations(state, writeNodeIdsExceedingQueueLatencyThreshold, currentTimeMillis);
        if (writeNodeIdsExceedingQueueLatencyThreshold.isEmpty()) {
            logger.trace("No hot-spotting write nodes detected");
            return;
        }
        if (!haveWriteNodesBelowQueueLatencyThreshold) {
            logger.debug("Nodes [{}] are above the queue latency threshold, but there are no write nodes below the threshold. Cannot rebalance shards.", (Object)WriteLoadConstraintMonitor.nodeSummary(writeNodeIdsExceedingQueueLatencyThreshold));
            return;
        }
        long timeSinceLastRerouteMillis = currentTimeMillis - this.lastRerouteTimeMillis;
        boolean haveCalledRerouteRecently = timeSinceLastRerouteMillis < this.writeLoadConstraintSettings.getMinimumRerouteInterval().millis();
        Set<String> newHotspotNodes = Sets.difference(writeNodeIdsExceedingQueueLatencyThreshold, lastHotspotNodes);
        if (!haveCalledRerouteRecently || !newHotspotNodes.isEmpty()) {
            if (logger.isDebugEnabled()) {
                logger.debug("Nodes [{}] are hot-spotting, of {} total ingest nodes. Reroute for hot-spotting {}. Previously hot-spotting nodes are [{}]. The write thread pool queue latency threshold is [{}]. Triggering reroute.\n", (Object)WriteLoadConstraintMonitor.nodeSummary(writeNodeIdsExceedingQueueLatencyThreshold), (Object)totalIngestNodes, this.lastRerouteTimeMillis == 0L ? "has never previously been called" : "was last called [" + String.valueOf(TimeValue.timeValueMillis(timeSinceLastRerouteMillis)) + "] ago", (Object)WriteLoadConstraintMonitor.nodeSummary(lastHotspotNodes), (Object)this.writeLoadConstraintSettings.getQueueLatencyThreshold());
            }
            String reason = "hot-spotting detected by write load constraint monitor";
            this.rerouteService.reroute("hot-spotting detected by write load constraint monitor", Priority.NORMAL, ActionListener.wrap(ignored -> logger.trace("{} reroute successful", (Object)"hot-spotting detected by write load constraint monitor"), e -> logger.debug(() -> Strings.format("reroute failed, reason: %s", "hot-spotting detected by write load constraint monitor"), (Throwable)e)));
            this.lastRerouteTimeMillis = this.currentTimeMillisSupplier.getAsLong();
            this.recordHotspotStartTimes(newHotspotNodes, currentTimeMillis);
        } else {
            logger.debug("Not calling reroute because we called reroute [{}] ago and there are no new hot spots", (Object)TimeValue.timeValueMillis(timeSinceLastRerouteMillis));
        }
    }

    private void recordHotspotStartTimes(Set<String> nodeIds, long startTimestamp) {
        for (String nodeId : nodeIds) {
            this.hotspotNodeStartTimes.put(nodeId, startTimestamp);
        }
        this.hotspotNodesCount.set(this.hotspotNodeStartTimes.size());
    }

    private Set<String> recordHotspotDurations(ClusterState state, Set<String> currentHotspotNodes, long hotspotEndTime) {
        if (state.term() != this.hotspotNodeStartTimesLastTerm || !state.nodes().isLocalNodeElectedMaster()) {
            this.hotspotNodeStartTimesLastTerm = state.term();
            this.hotspotNodeStartTimes.clear();
        }
        Set<String> lastHotspotNodes = this.hotspotNodeStartTimes.keySet();
        Set<String> staleHotspotNodes = Sets.difference(lastHotspotNodes, currentHotspotNodes);
        for (String nodeId : staleHotspotNodes) {
            assert (this.hotspotNodeStartTimes.containsKey(nodeId)) : "Map should contain key from its own subset";
            long hotspotStartTime = this.hotspotNodeStartTimes.remove(nodeId);
            long hotspotDuration = hotspotEndTime - hotspotStartTime;
            assert (hotspotDuration >= 0L) : "hotspot duration should always be non-negative";
            this.hotspotDurationHistogram.record((double)hotspotDuration / 1000.0);
        }
        this.hotspotNodesCount.set(this.hotspotNodeStartTimes.size());
        return lastHotspotNodes;
    }

    private List<LongWithAttributes> getHotspotNodesCount() {
        long hotspotCount = this.hotspotNodesCount.getAndSet(-1L);
        if (hotspotCount >= 0L) {
            return List.of(new LongWithAttributes(hotspotCount));
        }
        return List.of();
    }

    private static String nodeSummary(Set<String> nodeIds) {
        if (!nodeIds.isEmpty() && nodeIds.size() <= 3) {
            return "[" + String.join((CharSequence)", ", nodeIds) + "]";
        }
        return nodeIds.size() + " nodes";
    }
}

