"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useAllExceptionLists = void 0;
var _react = require("react");
var _user_privileges = require("../../../common/components/user_privileges");
var _api = require("../../../detection_engine/rule_management/api/api");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Hook for preparing exception lists table info. For now, we need to do a table scan
 * of all rules to figure out which exception lists are used in what rules. This is very
 * slow, however, there is an issue open that would push all this work to Kiaban/ES and
 * speed things up a ton - https://github.com/elastic/kibana/issues/85173
 *
 * @param exceptionLists ExceptionListSchema(s) to evaluate
 *
 */
const useAllExceptionLists = ({
  exceptionLists
}) => {
  const [loading, setLoading] = (0, _react.useState)(true);
  const [exceptions, setExceptions] = (0, _react.useState)([]);
  const [exceptionsListsInfo, setExceptionsListInfo] = (0, _react.useState)({});
  const {
    read: canReadRules
  } = (0, _user_privileges.useUserPrivileges)().rulesPrivileges;
  const handleExceptionsInfo = (0, _react.useCallback)(rules => {
    const listsSkeleton = exceptionLists.reduce((acc, {
      id,
      ...rest
    }) => {
      acc[id] = {
        ...rest,
        id,
        rules: []
      };
      return acc;
    }, {});
    return rules.reduce((acc, rule) => {
      const ruleExceptionLists = rule.exceptions_list;
      if (ruleExceptionLists != null && ruleExceptionLists.length > 0) {
        ruleExceptionLists.forEach(ex => {
          const list = acc[ex.id];
          if (list != null) {
            acc[ex.id] = {
              ...list,
              rules: [...list.rules, rule]
            };
          }
        });
      }
      return acc;
    }, listsSkeleton);
  }, [exceptionLists]);
  (0, _react.useEffect)(() => {
    let isSubscribed = true;
    const abortCtrl = new AbortController();
    const fetchData = async () => {
      if (exceptionLists.length === 0 && isSubscribed) {
        setLoading(false);
        setExceptions([]);
        setExceptionsListInfo({});
        return;
      }
      try {
        setLoading(true);
        const rules = [];
        if (canReadRules) {
          const {
            data: rulesResponse
          } = await (0, _api.fetchRules)({
            pagination: {
              page: 1,
              perPage: 10000
            },
            signal: abortCtrl.signal
          });
          rules.push(...rulesResponse);
        }
        const updatedLists = handleExceptionsInfo(rules);
        const lists = Object.keys(updatedLists).map(listKey => updatedLists[listKey]);
        setExceptions(lists);
        setExceptionsListInfo(updatedLists);
        if (isSubscribed) {
          setLoading(false);
        }
      } catch (error) {
        if (isSubscribed) {
          setLoading(false);
        }
      }
    };
    fetchData();
    return () => {
      isSubscribed = false;
      abortCtrl.abort();
    };
  }, [canReadRules, exceptionLists.length, handleExceptionsInfo]);
  return [loading, exceptions, exceptionsListsInfo];
};
exports.useAllExceptionLists = useAllExceptionLists;