"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ResourceIdentifier = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _splunk = require("./splunk");
var _types = require("../rules/resources/types");
var _qradar = require("./qradar");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/** Currently resource identification is only needed for Splunk since this for Qradar we identify resources by LLM */
const identifiers = {
  splunk: _splunk.splResourceIdentifier,
  qradar: _qradar.qradarResourceIdentifier
};

// Type for a class that extends the ResourceIdentifier abstract class

class ResourceIdentifier {
  constructor(vendor) {
    (0, _defineProperty2.default)(this, "identifier", void 0);
    this.vendor = vendor;
    // The constructor may need query_language as an argument for other vendors
    if (!(0, _types.isResourceSupportedVendor)(this.vendor)) {
      throw new Error(`Resource identification is not supported for vendor: ${this.vendor}`);
    }
    this.identifier = identifiers[this.vendor];
  }
  fromQuery(query) {
    return this.identifier(query);
  }
  async fromOriginals(originalItem) {
    const lookups = new Set();
    const macros = new Set();
    for (const item of originalItem) {
      const resources = await this.fromOriginal(item);
      resources.forEach(resource => {
        if (resource.type === 'macro') {
          macros.add(resource.name);
        } else if (resource.type === 'lookup') {
          lookups.add(resource.name);
        }
      });
    }
    return [...Array.from(macros).map(name => ({
      type: 'macro',
      name
    })), ...Array.from(lookups).map(name => ({
      type: 'lookup',
      name
    }))];
  }
  async fromResource(resource) {
    if (resource.type === 'macro' && resource.content && this.vendor === 'splunk') {
      return this.identifier(resource.content);
    }
    return Promise.resolve([]);
  }
  async fromResources(resources) {
    if (this.vendor !== 'splunk') {
      return [];
    }
    const lookups = new Set();
    const macros = new Set();
    for (const resource of resources) {
      const resourcesFromResource = await this.fromResource(resource);
      resourcesFromResource.forEach(identifiedResource => {
        if (identifiedResource.type === 'macro') {
          macros.add(identifiedResource.name);
        } else if (identifiedResource.type === 'lookup') {
          lookups.add(identifiedResource.name);
        }
      });
    }
    return [...Array.from(macros).map(name => ({
      type: 'macro',
      name
    })), ...Array.from(lookups).map(name => ({
      type: 'lookup',
      name
    }))];
  }
}
exports.ResourceIdentifier = ResourceIdentifier;