"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initIndices = exports.initDataFor = exports.calculateShardValues = void 0;
exports.mutateAggsTimesTree = mutateAggsTimesTree;
exports.mutateSearchTimesTree = mutateSearchTimesTree;
exports.normalize = void 0;
var _lodash = require("lodash");
var _function = require("fp-ts/function");
var _unsafe_utils = require("./unsafe_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Functions prefixed with "mutate" change values by reference. Be careful when using these!
 */
function mutateAggsTimesTree(shard) {
  if (shard.aggregations == null) {
    shard.time = 0;
  }
  let shardTime = 0;
  for (const agg of shard.aggregations) {
    const totalTime = (0, _unsafe_utils.calcTimes)([agg]);
    shardTime += totalTime;
  }
  for (const agg of shard.aggregations) {
    (0, _unsafe_utils.initTree)([agg], shardTime);
    // To make this data structure consistent with that of search we
    // mark each aggregation as it's own tree root.
    agg.treeRoot = agg;
  }
  shard.time = shardTime;
}
function mutateSearchTimesTree(shard) {
  if (shard.searches == null) {
    shard.time = 0;
  }
  shard.rewrite_time = 0;
  let shardTime = 0;
  for (const search of shard.searches) {
    shard.rewrite_time += search.rewrite_time;
    const totalTime = (0, _unsafe_utils.calcTimes)(search.query);
    shardTime += totalTime;
    (0, _unsafe_utils.initTree)(search.query, totalTime);
    search.treeRoot = search.query[0];
    // Remove this object.
    search.query = null;
  }
  shard.time = shardTime;
}
const initShards = data => data.map(s => {
  const idMatch = s.id.match(/\[([^\]\[]*?)\]/g) || [];
  const ids = idMatch.map(id => {
    return id.replace('[', '').replace(']', '');
  });
  return {
    ...s,
    id: ids,
    time: 0,
    color: '',
    relative: 0
  };
});
const calculateShardValues = target => data => {
  const mutateTimesTree = target === 'searches' ? mutateSearchTimesTree : target === 'aggregations' ? mutateAggsTimesTree : null;
  if (mutateTimesTree) {
    for (const shard of data) {
      mutateTimesTree(shard);
    }
  }
  return data;
};
exports.calculateShardValues = calculateShardValues;
const initIndices = data => {
  const indices = {};
  for (const shard of data) {
    if (!indices[shard.id[1]]) {
      indices[shard.id[1]] = {
        shards: [],
        time: 0,
        name: shard.id[1],
        visible: false
      };
    }
    indices[shard.id[1]].shards.push(shard);
    indices[shard.id[1]].time += shard.time;
  }
  return indices;
};
exports.initIndices = initIndices;
const normalize = target => data => {
  (0, _unsafe_utils.normalizeIndices)(data, target);
  return data;
};
exports.normalize = normalize;
const initDataFor = target => (0, _function.flow)(_lodash.cloneDeep, initShards, calculateShardValues(target), initIndices, normalize(target), _unsafe_utils.sortIndices);
exports.initDataFor = initDataFor;