"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateUserElasticsearchQuery = exports.formatElasticsearchQueryString = exports.elasticsearchQueryString = exports.elasticsearchQueryObject = exports.disableExecuteQuery = void 0;
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const USER_QUERY_PLACEHOLDER_MISSING_ERROR = _i18n.i18n.translate('xpack.searchPlayground.userQuery.errors.queryPlaceholder', {
  defaultMessage: 'Elasticsearch query must contain "{query}"',
  values: {
    query: '{query}'
  }
});
const USER_QUERY_CANNOT_BE_EMPTY_ERROR = _i18n.i18n.translate('xpack.searchPlayground.userQuery.errors.queryCannotBeEmpty', {
  defaultMessage: 'Elasticsearch query cannot be empty'
});
const USER_QUERY_MUST_BE_VALID_JSON_ERROR = _i18n.i18n.translate('xpack.searchPlayground.userQuery.errors.queryMustBeValidJson', {
  defaultMessage: 'Elasticsearch query must be valid JSON'
});
const validateUserElasticsearchQuery = userQuery => {
  if (userQuery === null || userQuery === undefined || typeof userQuery !== 'string') return undefined;
  if (userQuery.trim().length === 0) {
    return {
      type: 'required',
      message: USER_QUERY_CANNOT_BE_EMPTY_ERROR,
      types: {
        value: USER_QUERY_PLACEHOLDER_MISSING_ERROR,
        required: USER_QUERY_CANNOT_BE_EMPTY_ERROR
      }
    };
  }
  if (!userQuery.includes('{query}')) {
    return {
      type: 'value',
      message: USER_QUERY_PLACEHOLDER_MISSING_ERROR
    };
  }
  try {
    JSON.parse(userQuery);
  } catch (e) {
    return {
      type: 'validate',
      message: USER_QUERY_MUST_BE_VALID_JSON_ERROR
    }; // return query must be valid JSON error
  }
  return undefined;
};
exports.validateUserElasticsearchQuery = validateUserElasticsearchQuery;
const disableExecuteQuery = (validElasticsearchQuery, query) => {
  return !validElasticsearchQuery || !query || query.trim().length === 0;
};
exports.disableExecuteQuery = disableExecuteQuery;
const elasticsearchQueryString = (elasticsearchQuery, userElasticsearchQuery, userElasticsearchQueryError) => {
  if (!userElasticsearchQuery) {
    return JSON.stringify(elasticsearchQuery);
  }
  if (userElasticsearchQueryError === undefined) {
    return userElasticsearchQuery;
  }
  return JSON.stringify(elasticsearchQuery);
};
exports.elasticsearchQueryString = elasticsearchQueryString;
const elasticsearchQueryObject = (elasticsearchQuery, userElasticsearchQuery, userElasticsearchQueryError) => {
  if (!userElasticsearchQuery) {
    return elasticsearchQuery;
  }
  if (userElasticsearchQueryError === undefined) {
    return JSON.parse(userElasticsearchQuery);
  }
  return elasticsearchQuery;
};
exports.elasticsearchQueryObject = elasticsearchQueryObject;
const formatElasticsearchQueryString = elasticsearchQuery => JSON.stringify(elasticsearchQuery, null, 2);
exports.formatElasticsearchQueryString = formatElasticsearchQueryString;