"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseSecurityLabsArtifactName = exports.parseArtifactName = exports.isValidSecurityLabsVersion = exports.getSecurityLabsArtifactName = exports.getResourceTypeFromArtifactName = exports.getArtifactName = exports.DEFAULT_ELSER = void 0;
var _is_default_inference_endpoint = require("./is_default_inference_endpoint");
var _product = require("./product");
var _resource_type = require("./resource_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const allowedProductNames = Object.values(_product.DocumentationProduct);
const DEFAULT_ELSER = exports.DEFAULT_ELSER = '.elser-2-elasticsearch';

/**
 * Date version pattern for Security Labs artifacts (YYYY.MM.DD)
 */
const SECURITY_LABS_VERSION_PATTERN = /^\d{4}\.\d{2}\.\d{2}$/;
const getArtifactName = ({
  productName,
  productVersion,
  excludeExtension = false,
  inferenceId
}) => {
  const ext = excludeExtension ? '' : '.zip';
  return `kb-product-doc-${productName}-${productVersion}${inferenceId && !(0, _is_default_inference_endpoint.isImpliedDefaultElserInferenceId)(inferenceId) ? `--${inferenceId}` : ''}${ext}`.toLowerCase();
};
exports.getArtifactName = getArtifactName;
const parseArtifactName = artifactName => {
  // drop ".zip" (if any)
  let name = artifactName.endsWith('.zip') ? artifactName.slice(0, -4) : artifactName;

  // pull off the final  "--<inferenceId>" (if present)
  let inferenceId;
  const lastDashDash = name.lastIndexOf('--');
  if (lastDashDash !== -1) {
    inferenceId = name.slice(lastDashDash + 2);
    name = name.slice(0, lastDashDash); // strip it for the base match
  }

  // match the main pattern kb-product-doc-<product>-<version>
  const match = name.match(/^kb-product-doc-([a-z]+)-([0-9]+\.[0-9]+)$/);
  if (!match) return;
  const productName = match[1].toLowerCase();
  const productVersion = match[2].toLowerCase();
  if (!allowedProductNames.includes(productName)) return;
  return {
    productName,
    productVersion,
    ...(inferenceId ? {
      inferenceId
    } : {})
  };
};

/**
 * Generates the artifact name for Security Labs content.
 * Format: security-labs-{version}[--{inferenceId}].zip
 * Version uses date format: YYYY.MM.DD
 */
exports.parseArtifactName = parseArtifactName;
const getSecurityLabsArtifactName = ({
  version,
  excludeExtension = false,
  inferenceId
}) => {
  const ext = excludeExtension ? '' : '.zip';
  return `security-labs-${version}${inferenceId && !(0, _is_default_inference_endpoint.isImpliedDefaultElserInferenceId)(inferenceId) ? `--${inferenceId}` : ''}${ext}`.toLowerCase();
};

/**
 * Parses a Security Labs artifact name to extract version and optional inference ID.
 */
exports.getSecurityLabsArtifactName = getSecurityLabsArtifactName;
const parseSecurityLabsArtifactName = artifactName => {
  // drop ".zip" (if any)
  let name = artifactName.endsWith('.zip') ? artifactName.slice(0, -4) : artifactName;

  // pull off the final "--<inferenceId>" (if present)
  let inferenceId;
  const lastDashDash = name.lastIndexOf('--');
  if (lastDashDash !== -1) {
    inferenceId = name.slice(lastDashDash + 2);
    name = name.slice(0, lastDashDash);
  }

  // match the pattern security-labs-<version>
  const match = name.match(/^security-labs-(\d{4}\.\d{2}\.\d{2})$/);
  if (!match) return;
  const version = match[1];
  return {
    version,
    resourceType: _resource_type.ResourceTypes.securityLabs,
    ...(inferenceId ? {
      inferenceId
    } : {})
  };
};

/**
 * Determines the resource type from an artifact name.
 */
exports.parseSecurityLabsArtifactName = parseSecurityLabsArtifactName;
const getResourceTypeFromArtifactName = artifactName => {
  if (parseArtifactName(artifactName)) {
    return _resource_type.ResourceTypes.productDoc;
  }
  if (parseSecurityLabsArtifactName(artifactName)) {
    return _resource_type.ResourceTypes.securityLabs;
  }
  return undefined;
};

/**
 * Validates a Security Labs version string (YYYY.MM.DD format).
 */
exports.getResourceTypeFromArtifactName = getResourceTypeFromArtifactName;
const isValidSecurityLabsVersion = version => {
  return SECURITY_LABS_VERSION_PATTERN.test(version);
};
exports.isValidSecurityLabsVersion = isValidSecurityLabsVersion;