"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useProductDocStatus = useProductDocStatus;
var _reactQuery = require("@kbn/react-query");
var _inferenceCommon = require("@kbn/inference-common");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const POLLING_INTERVAL_MS = 5000; // Poll every 5 seconds during installation/uninstallation

const isProductDocsStatus = value => {
  return value != null && typeof value === 'object' && 'overall' in value;
};

/**
 * Hook to fetch the installation status of product documentation.
 * Automatically polls when installation or uninstallation is in progress.
 */
function useProductDocStatus(productDocBase, options = {}) {
  const {
    inferenceId = _inferenceCommon.defaultInferenceEndpoints.ELSER,
    resourceType
  } = options;
  const {
    isLoading,
    isError,
    isSuccess,
    isRefetching,
    data,
    refetch
  } = (0, _reactQuery.useQuery)({
    queryKey: [_constants.REACT_QUERY_KEYS.GET_PRODUCT_DOC_STATUS, inferenceId, resourceType],
    queryFn: async () => {
      // The response shape depends on resourceType (product docs vs security labs)
      return productDocBase.installation.getStatus({
        inferenceId,
        resourceType
      });
    },
    keepPreviousData: false,
    refetchOnWindowFocus: false,
    // Poll when installation or uninstallation is in progress
    refetchInterval: queryData => {
      const status = isProductDocsStatus(queryData) ? queryData.overall : queryData === null || queryData === void 0 ? void 0 : queryData.status;
      if (status === 'installing' || status === 'uninstalling') {
        return POLLING_INTERVAL_MS;
      }
      return false;
    }
  });
  return {
    status: data,
    refetch,
    isLoading,
    isRefetching,
    isSuccess,
    isError
  };
}