"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getMcpToolType = void 0;
exports.listMcpTools = listMcpTools;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _jsonSchemaToZod = require("@n8n/json-schema-to-zod");
var _schemas = require("./schemas");
var _validate_configuration = require("./validate_configuration");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Lists available tools from an MCP connector by calling the listTools subAction.
 */
async function listMcpTools({
  actions,
  request,
  connectorId
}) {
  const actionsClient = await actions.getActionsClientWithRequest(request);
  const result = await actionsClient.execute({
    actionId: connectorId,
    params: {
      subAction: 'listTools',
      subActionParams: {}
    }
  });
  if (result.status === 'error') {
    throw new Error(result.message || 'Failed to list MCP tools');
  }
  return result.data;
}

/**
 * Retrieves the input schema for a specific MCP tool by calling listTools on the connector.
 * Returns undefined if the connector or tool is not found.
 */
async function getMcpToolInputSchema({
  actions,
  request,
  connectorId,
  toolName
}) {
  try {
    const {
      tools
    } = await listMcpTools({
      actions,
      request,
      connectorId
    });
    const tool = tools.find(t => t.name === toolName);
    return tool === null || tool === void 0 ? void 0 : tool.inputSchema;
  } catch (error) {
    // Connector not found or other error - return undefined so getSchema will throw
    return undefined;
  }
}

/**
 * Discriminated union for MCP tool execution results
 */

/**
 * Executes an MCP tool via the connector's callTool sub-action
 */
async function executeMcpTool({
  actions,
  request,
  connectorId,
  toolName,
  toolArguments
}) {
  const actionsClient = await actions.getActionsClientWithRequest(request);
  const result = await actionsClient.execute({
    actionId: connectorId,
    params: {
      subAction: 'callTool',
      subActionParams: {
        name: toolName,
        arguments: toolArguments
      }
    }
  });
  if (result.status === 'error') {
    return {
      isError: true,
      content: result.message || 'MCP tool execution failed'
    };
  }
  return {
    isError: false,
    content: result.data
  };
}

/**
 * MCP Tool Type for Agent Builder.
 *
 * An MCP tool maps 1:1 to a tool provided by an MCP server, connected via an MCP Stack Connector.
 *
 * Architecture:
 * - connector_id: References the MCP Stack Connector that connects to the MCP server
 * - tool_name: The name of the tool on the MCP server to invoke
 * - Input schema: Retrieved by calling listTools on the MCP connector
 * - Execution: Calls the connector's callTool sub-action with the tool name and arguments
 */
const getMcpToolType = ({
  actions
}) => {
  return {
    toolType: _onechatCommon.ToolType.mcp,
    getDynamicProps: (config, {
      request
    }) => {
      return {
        getHandler: () => {
          return async (params, context) => {
            const {
              logger
            } = context;
            logger.debug(`Executing MCP tool: connector=${config.connector_id}, tool=${config.tool_name}`);
            try {
              const result = await executeMcpTool({
                actions,
                request,
                connectorId: config.connector_id,
                toolName: config.tool_name,
                toolArguments: params
              });
              if (result.isError) {
                return {
                  results: [{
                    type: _onechatCommon.ToolResultType.error,
                    data: {
                      message: String(result.content)
                    }
                  }]
                };
              }
              return {
                results: [{
                  type: _onechatCommon.ToolResultType.other,
                  // MCP tool results are dynamic - content type varies per tool.
                  // Cast is acceptable as ToolResultType.other handles arbitrary data.
                  data: result.content
                }]
              };
            } catch (error) {
              logger.error(`MCP tool execution failed: ${error}`);
              return {
                results: [{
                  type: _onechatCommon.ToolResultType.error,
                  data: {
                    message: `Failed to execute MCP tool: ${error instanceof Error ? error.message : String(error)}`
                  }
                }]
              };
            }
          };
        },
        getSchema: async () => {
          // Retrieve input schema by calling listTools on the MCP connector
          const inputSchema = await getMcpToolInputSchema({
            actions,
            request,
            connectorId: config.connector_id,
            toolName: config.tool_name
          });
          if (inputSchema) {
            const zodSchema = (0, _jsonSchemaToZod.jsonSchemaToZod)(inputSchema);
            return zodSchema;
          }
          return _zod.z.object({});
        },
        getLlmDescription: ({
          description
        }) => {
          return `${description}

## Additional information
- This tool is provided by an MCP (Model Context Protocol) server
- Server tool name: '${config.tool_name}'
`;
        }
      };
    },
    createSchema: _schemas.configurationSchema,
    updateSchema: _schemas.configurationUpdateSchema,
    validateForCreate: async ({
      config,
      context: {
        request
      }
    }) => {
      await (0, _validate_configuration.validateConfig)({
        actions,
        request,
        config
      });
      return config;
    },
    validateForUpdate: async ({
      update,
      current,
      context: {
        request
      }
    }) => {
      const mergedConfig = {
        ...current,
        ...update
      };
      await (0, _validate_configuration.validateConfig)({
        actions,
        request,
        config: mergedConfig
      });
      return mergedConfig;
    },
    // Track execution health for MCP tools since they depend on external MCP servers
    trackHealth: true
  };
};
exports.getMcpToolType = getMcpToolType;