"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isRecoverableError = exports.convertError = void 0;
var _errors = require("@kbn/inference-common/src/chat_complete/errors");
var _agents = require("@kbn/onechat-common/agents");
var _errors2 = require("@kbn/onechat-common/base/errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const recoverableErrorCodes = [_agents.AgentExecutionErrorCode.toolNotFound, _agents.AgentExecutionErrorCode.toolValidationError, _agents.AgentExecutionErrorCode.emptyResponse];

/**
 * Converts an error which occurred during the execution of the agent to our error format,
 * leveraging the errors which are already processed by the inference plugin for some of them.
 * Also categorizes the error to identifiable error codes.
 */
const convertError = error => {
  if ((0, _errors.isToolNotFoundError)(error)) {
    return (0, _errors2.createAgentExecutionError)(error.message, _agents.AgentExecutionErrorCode.toolNotFound, {
      toolName: error.meta.name,
      toolArgs: error.meta.arguments
    });
  } else if ((0, _errors.isToolValidationError)(error)) {
    var _error$meta$name, _error$meta$arguments;
    return (0, _errors2.createAgentExecutionError)(error.message, _agents.AgentExecutionErrorCode.toolValidationError, {
      toolName: (_error$meta$name = error.meta.name) !== null && _error$meta$name !== void 0 ? _error$meta$name : 'unknown',
      toolArgs: (_error$meta$arguments = error.meta.arguments) !== null && _error$meta$arguments !== void 0 ? _error$meta$arguments : '',
      validationError: error.meta.errorsText
    });
  } else if ((0, _errors.isContextLengthExceededError)(error)) {
    return (0, _errors2.createAgentExecutionError)(error.message, _agents.AgentExecutionErrorCode.contextLengthExceeded, {});
  } else {
    return (0, _errors2.createAgentExecutionError)(error.message, _agents.AgentExecutionErrorCode.unknownError, {});
  }
};
exports.convertError = convertError;
const isRecoverableError = error => {
  if (!(0, _errors2.isAgentExecutionError)(error)) {
    return false;
  }
  return recoverableErrorCodes.includes(error.meta.errCode);
};
exports.isRecoverableError = isRecoverableError;