"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.structuredOutputSchema = exports.createAnswerAgentStructured = void 0;
var _zod = require("@kbn/zod");
var _langchain = require("@kbn/onechat-genai-utils/langchain");
var _errors = require("../utils/errors");
var _actions = require("./actions");
var _prompts = require("./prompts");
var _i18n = require("./i18n");
var _constants = require("./constants");
var _action_utils = require("./action_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const structuredOutputSchema = exports.structuredOutputSchema = _zod.z.object({
  response: _zod.z.string().describe("The response to the user's query"),
  data: _zod.z.record(_zod.z.unknown()).optional().describe('Optional structured data to include in the response')
});

/**
 * Structured output answer agent with structured error handling.
 * This agent uses structured output mode and returns structured error responses.
 */
const createAnswerAgentStructured = ({
  chatModel,
  configuration,
  capabilities,
  events,
  outputSchema,
  attachmentTypes
}) => {
  return async state => {
    if (state.answerActions.length === 0 && state.errorCount === 0) {
      events.emit((0, _langchain.createReasoningEvent)((0, _i18n.getRandomAnsweringMessage)(), {
        transient: true
      }));
    }
    try {
      let schemaToUse = outputSchema ? outputSchema : structuredOutputSchema;

      // Add description to JSON Schema if it doesn't have one, for some reason without it this doesnt seem to work reliably
      if (!('description' in schemaToUse) && typeof schemaToUse === 'object' && schemaToUse !== null) {
        schemaToUse = {
          ...schemaToUse,
          description: "Use this structured format to respond to the user's request with the required data."
        };
      }
      const structuredModel = chatModel.withStructuredOutput(schemaToUse, {
        name: 'structured_answer'
      }).withConfig({
        tags: [_constants.tags.agent, _constants.tags.answerAgent]
      });
      const prompt = (0, _prompts.getStructuredAnswerPrompt)({
        customInstructions: configuration.answer.instructions,
        capabilities,
        initialMessages: state.initialMessages,
        actions: state.mainActions,
        answerActions: state.answerActions,
        attachmentTypes
      });
      const response = await structuredModel.invoke(prompt);
      const action = (0, _action_utils.processStructuredAnswerResponse)(response);
      return {
        answerActions: [action],
        errorCount: 0
      };
    } catch (error) {
      const executionError = (0, _errors.convertError)(error);
      if ((0, _errors.isRecoverableError)(executionError)) {
        return {
          answerActions: [(0, _actions.errorAction)(executionError)],
          errorCount: state.errorCount + 1
        };
      } else {
        throw executionError;
      }
    }
  };
};
exports.createAnswerAgentStructured = createAnswerAgentStructured;