"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.toInternalDefinition = exports.createBuiltinProviderFn = void 0;
var _onechatCommon = require("@kbn/onechat-common");
var _availability_cache = require("./availability_cache");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createBuiltinProviderFn = ({
  registry
}) => {
  const availabilityCache = new _availability_cache.AgentAvailabilityCache();
  return ({
    request
  }) => {
    return registryToProvider({
      registry,
      request,
      availabilityCache
    });
  };
};
exports.createBuiltinProviderFn = createBuiltinProviderFn;
const registryToProvider = ({
  registry,
  request,
  availabilityCache
}) => {
  return {
    id: 'builtin',
    readonly: true,
    has: agentId => {
      return registry.has(agentId);
    },
    get: agentId => {
      const definition = registry.get(agentId);
      if (!definition) {
        throw (0, _onechatCommon.createAgentNotFoundError)({
          agentId
        });
      }
      return toInternalDefinition({
        definition,
        availabilityCache
      });
    },
    list: opts => {
      const definitions = registry.list();
      return Promise.all(definitions.map(definition => toInternalDefinition({
        definition,
        availabilityCache
      })));
    }
  };
};
const toInternalDefinition = async ({
  definition,
  availabilityCache
}) => {
  return {
    ...definition,
    type: _onechatCommon.AgentType.chat,
    readonly: true,
    isAvailable: async ctx => {
      if (definition.availability) {
        return availabilityCache.getOrCompute(definition.id, definition.availability, ctx);
      } else {
        return {
          status: 'available'
        };
      }
    }
  };
};
exports.toInternalDefinition = toInternalDefinition;