"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AgentAvailabilityCache = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lruCache = require("lru-cache");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_TTL = 5 * 60 * 1000; // 5 minutes default TTL

class AgentAvailabilityCache {
  constructor() {
    (0, _defineProperty2.default)(this, "cache", new _lruCache.LRUCache({
      max: 1000,
      ttl: DEFAULT_TTL,
      allowStale: false,
      ttlAutopurge: false
    }));
  }
  /**
   * Get from cache, or recompute and store, then return.
   */
  async getOrCompute(agentId, config, context) {
    const cacheKey = getCacheKey(agentId, config, context);
    const cachedValue = cacheKey ? this.cache.get(cacheKey) : undefined;
    if (cachedValue !== undefined) {
      return cachedValue;
    }
    const newValue = await config.handler(context);
    if (cacheKey) {
      this.cache.set(cacheKey, newValue, {
        ttl: config.cacheTtl ? config.cacheTtl * 1000 : DEFAULT_TTL
      });
    }
    return newValue;
  }
  clear() {
    this.cache.clear();
  }

  /**
   * Check if a value is cached for the given tool and context.
   */
  has(agentId, config, context) {
    const cacheKey = getCacheKey(agentId, config, context);
    return cacheKey ? this.cache.has(cacheKey) : false;
  }
}
exports.AgentAvailabilityCache = AgentAvailabilityCache;
const getCacheKey = (agentId, config, context) => {
  switch (config.cacheMode) {
    case 'global':
      return `${agentId}`;
    case 'space':
      return `${agentId}||${context.spaceId}`;
    case 'none':
      return undefined;
  }
};