"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useMessageEditor = void 0;
var _react = require("react");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Creates an imperative handle for controlling MessageEditor
 *
 * @example
 * const editor = useMessageEditor();
 * editor.focus();
 * const content = editor.getContent();
 * if (editor.isEmpty) {
 *   // Submit button disabled
 * }
 *
 * <MessageEditor messageEditor={editor} />
 */
const useMessageEditor = () => {
  const ref = (0, _react.useRef)(null);
  const [isEmpty, setIsEmpty] = (0, _react.useState)(true);
  const syncIsEmpty = (0, _react.useCallback)(() => {
    if (!(ref !== null && ref !== void 0 && ref.current)) {
      return;
    }
    const nextIsEmpty = !ref.current.textContent || ref.current.textContent.trim() === '';
    if (nextIsEmpty) {
      // If current text content is empty clear innerHTML
      // This is required so the :empty pseudo-class gets reset and the placeholder is shown
      ref.current.innerHTML = '';
    }
    setIsEmpty(nextIsEmpty);
  }, []);
  const instance = (0, _react.useMemo)(() => ({
    _internal: {
      ref,
      onChange: () => {
        syncIsEmpty();
      }
    },
    focus: () => {
      var _ref$current;
      (_ref$current = ref.current) === null || _ref$current === void 0 ? void 0 : _ref$current.focus();
    },
    getContent: () => {
      var _ref$current$textCont, _ref$current2;
      return (_ref$current$textCont = (_ref$current2 = ref.current) === null || _ref$current2 === void 0 ? void 0 : _ref$current2.textContent) !== null && _ref$current$textCont !== void 0 ? _ref$current$textCont : '';
    },
    setContent: text => {
      if (ref.current) {
        ref.current.textContent = text;
        syncIsEmpty();
        // Set caret position at the end of the text
        const selection = window.getSelection();
        if (selection && ref.current.firstChild) {
          selection.setPosition(ref.current.firstChild, text.length);
        }
      }
    },
    clear: () => {
      if (ref.current) {
        ref.current.textContent = '';
        setIsEmpty(true);
      }
    },
    isEmpty
  }), [isEmpty, syncIsEmpty]);
  return instance;
};
exports.useMessageEditor = useMessageEditor;