"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAgentBuilderResourceAvailability = getAgentBuilderResourceAvailability;
var _aiAssistantCommon = require("@kbn/ai-assistant-common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Availability handler for Observability Agent Builder resources.
 * Gates availability to Observability or Classic solution spaces.
 * If spaces are unavailable, returns available.
 */
async function getAgentBuilderResourceAvailability({
  core,
  request,
  logger
}) {
  const [coreStart, pluginsStart] = await core.getStartServices();
  const isAiAgentsEnabled = await coreStart.featureFlags.getBooleanValue(_aiAssistantCommon.AI_AGENTS_FEATURE_FLAG, _aiAssistantCommon.AI_AGENTS_FEATURE_FLAG_DEFAULT);
  if (!isAiAgentsEnabled) {
    logger.debug(`AI agents are disabled (${_aiAssistantCommon.AI_AGENTS_FEATURE_FLAG}), skipping registration.`);
    return {
      status: 'unavailable',
      reason: `AI agents are disabled (${_aiAssistantCommon.AI_AGENTS_FEATURE_FLAG})`
    };
  }
  try {
    var _pluginsStart$spaces;
    const activeSpace = await ((_pluginsStart$spaces = pluginsStart.spaces) === null || _pluginsStart$spaces === void 0 ? void 0 : _pluginsStart$spaces.spacesService.getActiveSpace(request));
    const solution = activeSpace === null || activeSpace === void 0 ? void 0 : activeSpace.solution;
    const isAllowedSolution = !solution || solution === 'classic' || solution === 'oblt';
    if (!isAllowedSolution) {
      logger.debug('Observability agent builder resources are not available in this space, skipping registration.');
      return {
        status: 'unavailable',
        reason: 'Observability agent builder resources are not available in this space'
      };
    }
  } catch (error) {
    logger.debug('Spaces are unavailable, returning available for Observability agent builder resources.');
    logger.debug(error);
  }
  return {
    status: 'available'
  };
}