"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OBSERVABILITY_GET_HOSTS_TOOL_ID = void 0;
exports.createGetHostsTool = createGetHostsTool;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _get_agent_builder_resource_availability = require("../../utils/get_agent_builder_resource_availability");
var _tool_schemas = require("../../utils/tool_schemas");
var _handler = require("./handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OBSERVABILITY_GET_HOSTS_TOOL_ID = exports.OBSERVABILITY_GET_HOSTS_TOOL_ID = 'observability.get_hosts';
const DEFAULT_TIME_RANGE = {
  start: 'now-1h',
  end: 'now'
};
const DEFAULT_LIMIT = 20;
const MAX_LIMIT = 100;
const getHostsSchema = _zod.z.object({
  ...(0, _tool_schemas.timeRangeSchemaOptional)(DEFAULT_TIME_RANGE),
  limit: _zod.z.number().int().min(1).max(MAX_LIMIT).describe(`Maximum number of hosts to return. Defaults to ${DEFAULT_LIMIT}, maximum is ${MAX_LIMIT}.`).optional(),
  kqlFilter: _zod.z.string().optional().describe('Optional KQL filter to narrow down results. Examples: "service.name: frontend" (show only hosts running the frontend service), "host.name: web-*", or "cloud.provider: aws".')
});
function createGetHostsTool({
  core,
  logger,
  dataRegistry
}) {
  const toolDefinition = {
    id: OBSERVABILITY_GET_HOSTS_TOOL_ID,
    type: _onechatCommon.ToolType.builtin,
    description: `Retrieves a list of hosts with their infrastructure metrics (CPU, memory, disk, network). Use this tool to get an overview of host health and resource utilization.

When to use:
- Getting a high-level view of infrastructure health
- Identifying hosts with high CPU/memory usage or disk space issues
- Checking network throughput across hosts
- Answering questions like "which hosts are under heavy load?" or "what's the memory usage of my servers?"

Returns host names, metrics (CPU percentage, memory usage, disk space, network rx/tx), and metadata (OS name, cloud provider, IP address).`,
    schema: getHostsSchema,
    tags: ['observability', 'infrastructure', 'hosts', 'metrics'],
    availability: {
      cacheMode: 'space',
      handler: async ({
        request
      }) => {
        return (0, _get_agent_builder_resource_availability.getAgentBuilderResourceAvailability)({
          core,
          request,
          logger
        });
      }
    },
    handler: async ({
      start = DEFAULT_TIME_RANGE.start,
      end = DEFAULT_TIME_RANGE.end,
      limit = DEFAULT_LIMIT,
      kqlFilter
    }, {
      request
    }) => {
      try {
        const {
          hosts,
          total
        } = await (0, _handler.getToolHandler)({
          request,
          dataRegistry,
          start,
          end,
          limit,
          kqlFilter
        });
        return {
          results: [{
            type: _tool_result.ToolResultType.other,
            data: {
              hosts,
              total
            }
          }]
        };
      } catch (error) {
        logger.error(`Error fetching hosts: ${error.message}`);
        logger.debug(error);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: `Failed to fetch hosts: ${error.message}`,
              stack: error.stack
            }
          }]
        };
      }
    }
  };
  return toolDefinition;
}