"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getToolHandler = getToolHandler;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getToolHandler({
  core,
  plugins,
  mlClient,
  request,
  logger,
  jobIds = [],
  jobsLimit,
  rangeStart,
  rangeEnd
}) {
  var _plugins$ml;
  const [coreStart] = await core.getStartServices();
  const savedObjectsClient = coreStart.savedObjects.getScopedClient(request);
  const mlSystem = (_plugins$ml = plugins.ml) === null || _plugins$ml === void 0 ? void 0 : _plugins$ml.mlSystemProvider(request, savedObjectsClient);
  if (!mlSystem) {
    throw new Error('Machine Learning plugin is unavailable.');
  }
  const {
    jobs = []
  } = await mlClient.getJobs({
    job_id: jobIds.join(',')
  }).catch(error => {
    if (error.statusCode === 404) {
      return {
        jobs: []
      };
    }
    logger.error(`Error retrieving ML jobs: ${error.message}`);
    throw error;
  });
  return Promise.all(jobs.slice(0, jobsLimit).map(async job => {
    var _job$analysis_config, _job$datafeed_config, _job$analysis_config2, _job$analysis_config3;
    const topAnomalies = await getTopAnomalyRecords({
      mlSystem,
      jobId: job.job_id,
      start: rangeStart,
      end: rangeEnd
    });
    return {
      jobId: job.job_id,
      description: job.description,
      bucketSpan: (_job$analysis_config = job.analysis_config) === null || _job$analysis_config === void 0 ? void 0 : _job$analysis_config.bucket_span,
      datafeedIndices: (_job$datafeed_config = job.datafeed_config) === null || _job$datafeed_config === void 0 ? void 0 : _job$datafeed_config.indices,
      detectors: (_job$analysis_config2 = job.analysis_config) === null || _job$analysis_config2 === void 0 ? void 0 : (_job$analysis_config3 = _job$analysis_config2.detectors) === null || _job$analysis_config3 === void 0 ? void 0 : _job$analysis_config3.map(detector => ({
        description: detector.detector_description,
        function: detector.function,
        fieldName: detector.field_name
      })),
      topAnomalies
    };
  }));
}
async function getTopAnomalyRecords({
  mlSystem,
  jobId,
  start,
  end
}) {
  const response = await mlSystem.mlAnomalySearch({
    track_total_hits: false,
    size: 100,
    sort: [{
      record_score: {
        order: 'desc'
      }
    }],
    query: {
      bool: {
        filter: [{
          term: {
            job_id: jobId
          }
        }, {
          term: {
            result_type: 'record'
          }
        }, {
          term: {
            is_interim: false
          }
        }, {
          range: {
            timestamp: {
              gte: start,
              lte: end
            }
          }
        }]
      }
    },
    _source: ['timestamp', 'record_score', 'by_field_name', 'by_field_value', 'partition_field_name', 'partition_field_value', 'field_name', 'anomaly_score_explanation', 'typical', 'actual']
  }, [jobId]);
  const records = response.hits.hits.map(hit => hit._source).filter(record => record !== undefined);
  return records.map(record => ({
    timestamp: record.timestamp,
    anomalyScore: record.record_score,
    byFieldName: record.by_field_name,
    byFieldValue: record.by_field_value,
    partitionFieldName: record.partition_field_name,
    partitionFieldValue: record.partition_field_value,
    fieldName: record.field_name,
    anomalyScoreExplanation: record.anomaly_score_explanation,
    typicalValue: record.typical,
    actualValue: record.actual
  }));
}