"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OBSERVABILITY_GET_ALERTS_TOOL_ID = void 0;
exports.createGetAlertsTool = createGetAlertsTool;
exports.defaultFields = void 0;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _get_agent_builder_resource_availability = require("../../utils/get_agent_builder_resource_availability");
var _tool_schemas = require("../../utils/tool_schemas");
var _handler = require("./handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OBSERVABILITY_GET_ALERTS_TOOL_ID = exports.OBSERVABILITY_GET_ALERTS_TOOL_ID = 'observability.get_alerts';
const DEFAULT_TIME_RANGE = {
  start: 'now-1h',
  end: 'now'
};
const defaultFields = exports.defaultFields = ['@timestamp', 'kibana.alert.start', 'kibana.alert.end', 'kibana.alert.flapping', 'kibana.alert.group', 'kibana.alert.instance.id', 'kibana.alert.reason', 'kibana.alert.rule.category', 'kibana.alert.rule.name', 'kibana.alert.rule.tags', 'kibana.alert.start', 'kibana.alert.status', 'kibana.alert.time_range.gte', 'kibana.alert.time_range.lte', 'kibana.alert.workflow_status', 'tags',
// infra
'host.name', 'container.id', 'kubernetes.pod.name',
// APM
'processor.event', 'service.environment', 'service.name', 'service.node.name', 'transaction.type', 'transaction.name'];
const getAlertsSchema = _zod.z.object({
  ...(0, _tool_schemas.timeRangeSchemaOptional)(DEFAULT_TIME_RANGE),
  query: _zod.z.string().min(1).describe('Natural language query to guide relevant field selection.'),
  kqlFilter: _zod.z.string().optional().describe('Filter alerts by field:value pairs'),
  includeRecovered: _zod.z.boolean().optional().describe('Whether to include recovered/closed alerts. Defaults to false, which means only active alerts will be returned.')
});
function createGetAlertsTool({
  core,
  logger
}) {
  const toolDefinition = {
    id: OBSERVABILITY_GET_ALERTS_TOOL_ID,
    type: _onechatCommon.ToolType.builtin,
    description: `Retrieves Observability alerts within a specified time range. Supports filtering by status (active/recovered) and KQL queries to find specific alert instances.`,
    schema: getAlertsSchema,
    tags: ['observability', 'alerts'],
    availability: {
      cacheMode: 'space',
      handler: async ({
        request
      }) => {
        return (0, _get_agent_builder_resource_availability.getAgentBuilderResourceAvailability)({
          core,
          request,
          logger
        });
      }
    },
    handler: async ({
      start = DEFAULT_TIME_RANGE.start,
      end = DEFAULT_TIME_RANGE.end,
      kqlFilter,
      includeRecovered,
      query
    }, {
      request
    }) => {
      try {
        const {
          alerts,
          selectedFields,
          total
        } = await (0, _handler.getToolHandler)({
          core,
          request,
          logger,
          start,
          end,
          query,
          kqlFilter,
          includeRecovered
        });
        return {
          results: [{
            type: _tool_result.ToolResultType.other,
            data: {
              total,
              alerts,
              selectedFields: selectedFields.length === 0 ? defaultFields : selectedFields
            }
          }]
        };
      } catch (error) {
        logger.error(`Error fetching observability alerts: ${error.message}`);
        logger.debug(error);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: `Failed to fetch observability alerts: ${error.message}`,
              stack: error.stack
            }
          }]
        };
      }
    }
  };
  return toolDefinition;
}