"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchApmErrorContext = fetchApmErrorContext;
var _dedent = _interopRequireDefault(require("dedent"));
var _dsl_filters = require("../../../utils/dsl_filters");
var _handler = require("../../../tools/get_log_categories/handler");
var _get_logs_indices = require("../../../utils/get_logs_indices");
var _get_apm_indices = require("../../../utils/get_apm_indices");
var _time = require("../../../utils/time");
var _fetch_distributed_trace = require("./fetch_distributed_trace");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function fetchApmErrorContext({
  core,
  plugins,
  dataRegistry,
  request,
  errorId,
  serviceName,
  start,
  end,
  environment = '',
  logger
}) {
  var _errorDetails$transac, _errorDetails$transac2, _errorDetails$transac3, _errorDetails$error, _errorDetails$error$t;
  const parsedStart = (0, _time.parseDatemath)(start);
  const parsedEnd = (0, _time.parseDatemath)(end, {
    roundUp: true
  });
  const [coreStart] = await core.getStartServices();
  const esClient = coreStart.elasticsearch.client.asScoped(request);
  const errorDetails = await dataRegistry.getData('apmErrorDetails', {
    request,
    errorId,
    serviceName,
    start,
    end,
    serviceEnvironment: environment !== null && environment !== void 0 ? environment : ''
  });
  const traceId = (_errorDetails$transac = errorDetails === null || errorDetails === void 0 ? void 0 : (_errorDetails$transac2 = errorDetails.transaction) === null || _errorDetails$transac2 === void 0 ? void 0 : (_errorDetails$transac3 = _errorDetails$transac2.trace) === null || _errorDetails$transac3 === void 0 ? void 0 : _errorDetails$transac3.id) !== null && _errorDetails$transac !== void 0 ? _errorDetails$transac : errorDetails === null || errorDetails === void 0 ? void 0 : (_errorDetails$error = errorDetails.error) === null || _errorDetails$error === void 0 ? void 0 : (_errorDetails$error$t = _errorDetails$error.trace) === null || _errorDetails$error$t === void 0 ? void 0 : _errorDetails$error$t.id;
  const contextParts = [{
    name: 'ErrorDetails',
    start,
    end,
    handler: async () => errorDetails === null || errorDetails === void 0 ? void 0 : errorDetails.error
  }, {
    name: 'TransactionDetails',
    start,
    end,
    handler: async () => errorDetails === null || errorDetails === void 0 ? void 0 : errorDetails.transaction
  }, {
    name: 'DownstreamDependencies',
    start,
    end,
    handler: () => dataRegistry.getData('apmDownstreamDependencies', {
      request,
      serviceName,
      serviceEnvironment: environment !== null && environment !== void 0 ? environment : '',
      start,
      end
    })
  }];
  if (traceId) {
    const traceContextPromise = (async () => {
      const apmIndices = await (0, _get_apm_indices.getApmIndices)({
        core,
        plugins,
        logger
      });
      return (0, _fetch_distributed_trace.fetchDistributedTrace)({
        esClient,
        apmIndices,
        traceId,
        start: parsedStart,
        end: parsedEnd,
        logger
      });
    })();
    contextParts.push({
      name: 'TraceDocuments',
      start,
      end,
      handler: async () => {
        const {
          traceDocuments,
          isPartialTrace
        } = await traceContextPromise;
        return {
          isPartialTrace,
          documents: traceDocuments
        };
      }
    });
    contextParts.push({
      name: 'TraceServices',
      start,
      end,
      handler: async () => (await traceContextPromise).services
    });
    contextParts.push({
      name: 'TraceLogCategories',
      start,
      end,
      handler: async () => {
        const logsIndices = await (0, _get_logs_indices.getLogsIndices)({
          core,
          logger
        });
        const logCategories = await (0, _handler.getFilteredLogCategories)({
          esClient,
          logsIndices,
          boolQuery: {
            filter: [...(0, _dsl_filters.termFilter)('trace.id', traceId), {
              exists: {
                field: 'message'
              }
            }, {
              range: {
                '@timestamp': {
                  gte: parsedStart,
                  lte: parsedEnd
                }
              }
            }]
          },
          logger,
          categoryCount: 10,
          terms: {
            'trace.id': traceId
          }
        });
        return logCategories === null || logCategories === void 0 ? void 0 : logCategories.categories;
      }
    });
  }
  const results = await Promise.all(contextParts.map(async ({
    name,
    handler,
    start: partStart,
    end: partEnd
  }) => {
    try {
      const data = await handler();
      if (!data || Array.isArray(data) && data.length === 0) {
        return undefined;
      }
      return (0, _dedent.default)`<${name}>
          Time window: ${partStart} to ${partEnd}
          \`\`\`json
          ${JSON.stringify(data, null, 2)}
          \`\`\`
          </${name}>`;
    } catch (err) {
      logger.debug(`Error AI Insight: ${name} failed: ${err}`);
      return undefined;
    }
  }));
  return results.filter(Boolean).join('\n\n');
}