"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createLogAttachmentType = createLogAttachmentType;
var _zod = require("@kbn/zod");
var _dedent = _interopRequireDefault(require("dedent"));
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _common = require("../../common");
var _get_log_document_by_id = require("../routes/ai_insights/get_log_document_by_id");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const GET_LOG_DOCUMENT_TOOL_ID = 'get_log_document';
const logDataSchema = _zod.z.object({
  id: _zod.z.string(),
  index: _zod.z.string()
});
function createLogAttachmentType({
  core,
  logger,
  dataRegistry
}) {
  return {
    id: _common.OBSERVABILITY_LOG_ATTACHMENT_TYPE_ID,
    validate: input => {
      const parsed = logDataSchema.safeParse(input);
      if (parsed.success) {
        return {
          valid: true,
          data: parsed.data
        };
      }
      return {
        valid: false,
        error: parsed.error.message
      };
    },
    format: attachment => {
      const {
        index,
        id
      } = attachment.data;
      return {
        getRepresentation: () => ({
          type: 'text',
          value: `Observability Log ID: ${index}:${id}. Use the ${GET_LOG_DOCUMENT_TOOL_ID} tool to fetch full log document.`
        }),
        getBoundedTools: () => [{
          id: GET_LOG_DOCUMENT_TOOL_ID,
          type: _onechatCommon.ToolType.builtin,
          description: `Fetch the log document for ${index}:${id}.`,
          schema: _zod.z.object({}),
          handler: async (_args, context) => {
            try {
              const [coreStart] = await core.getStartServices();
              const esClient = coreStart.elasticsearch.client.asScoped(context.request);
              const logEntry = await (0, _get_log_document_by_id.getLogDocumentById)({
                esClient: esClient.asCurrentUser,
                index,
                id
              });
              if (!logEntry) {
                return {
                  results: [{
                    type: _tool_result.ToolResultType.error,
                    data: {
                      message: `Log document not found for ${index}:${id}`
                    }
                  }]
                };
              }
              return {
                results: [{
                  type: _tool_result.ToolResultType.other,
                  data: logEntry
                }]
              };
            } catch (error) {
              logger.error(`Failed to fetch log document for attachment: ${error === null || error === void 0 ? void 0 : error.message}`);
              logger.debug(error);
              return {
                results: [{
                  type: _tool_result.ToolResultType.error,
                  data: {
                    message: `Failed to fetch log document: ${error.message}`,
                    stack: error.stack
                  }
                }]
              };
            }
          }
        }]
      };
    },
    getTools: () => [],
    getAgentDescription: () => (0, _dedent.default)(`An Observability Log attachment. The log ID is provided - use the ${GET_LOG_DOCUMENT_TOOL_ID} tool to fetch the full log information.`)
  };
}