"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createErrorAttachmentType = createErrorAttachmentType;
var _zod = require("@kbn/zod");
var _dedent = _interopRequireDefault(require("dedent"));
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _common = require("../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const GET_ERROR_DETAILS_TOOL_ID = 'get_error_details';
const errorDataSchema = _zod.z.object({
  errorId: _zod.z.string(),
  serviceName: _zod.z.string().optional(),
  environment: _zod.z.string().nullable().optional(),
  start: _zod.z.string().optional(),
  end: _zod.z.string().optional()
});
function createErrorAttachmentType({
  core,
  logger,
  dataRegistry
}) {
  return {
    id: _common.OBSERVABILITY_ERROR_ATTACHMENT_TYPE_ID,
    validate: input => {
      const parsed = errorDataSchema.safeParse(input);
      if (parsed.success) {
        return {
          valid: true,
          data: parsed.data
        };
      }
      return {
        valid: false,
        error: parsed.error.message
      };
    },
    format: attachment => {
      const {
        errorId,
        serviceName = '',
        environment = '',
        start = '',
        end = ''
      } = attachment.data;
      return {
        getRepresentation: () => ({
          type: 'text',
          value: `Observability APM Error ID: ${errorId}. Use the ${GET_ERROR_DETAILS_TOOL_ID} tool to fetch full error information.`
        }),
        getBoundedTools: () => [{
          id: GET_ERROR_DETAILS_TOOL_ID,
          type: _onechatCommon.ToolType.builtin,
          description: `Fetch the full error information for error ${errorId}.`,
          schema: _zod.z.object({}),
          handler: async (_args, context) => {
            try {
              const errorDetails = await dataRegistry.getData('apmErrorDetails', {
                request: context.request,
                errorId,
                serviceName,
                start,
                end,
                serviceEnvironment: environment !== null && environment !== void 0 ? environment : ''
              });
              return {
                results: [{
                  type: _tool_result.ToolResultType.other,
                  data: errorDetails
                }]
              };
            } catch (error) {
              logger.error(`Failed to fetch error details for attachment: ${error === null || error === void 0 ? void 0 : error.message}`);
              logger.debug(error);
              return {
                results: [{
                  type: _tool_result.ToolResultType.error,
                  data: {
                    message: `Failed to fetch error details: ${error.message}`,
                    stack: error.stack
                  }
                }]
              };
            }
          }
        }]
      };
    },
    getTools: () => [],
    getAgentDescription: () => (0, _dedent.default)(`An Observability APM error attachment. The error ID is provided - use the ${GET_ERROR_DETAILS_TOOL_ID} tool to fetch the full error information.`)
  };
}