"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.annotationRoutes = annotationRoutes;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _i18n = require("@kbn/i18n");
var _app = require("../../common/constants/app");
var _check_annotations = require("../lib/check_annotations");
var _annotation_service = require("../models/annotation_service");
var _error_wrapper = require("../client/error_wrapper");
var _annotations_schema = require("./schemas/annotations_schema");
var _annotations = require("../../common/constants/annotations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getAnnotationsFeatureUnavailableErrorMessage() {
  return _boom.default.badRequest(_i18n.i18n.translate('xpack.ml.routes.annotations.annotationsFeatureUnavailableErrorMessage', {
    defaultMessage: 'Index and aliases required for the annotations feature have not been' + ' created or are not accessible for the current user.'
  }));
}
/**
 * Routes for annotations
 */
function annotationRoutes({
  router,
  routeGuard
}, securityPlugin) {
  /**
   * @apiGroup Annotations
   */
  router.versioned.post({
    path: `${_app.ML_INTERNAL_BASE_PATH}/annotations`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canGetAnnotations']
      }
    },
    summary: 'Gets annotations',
    description: 'Gets annotations.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: _annotations_schema.getAnnotationsSchema
      },
      response: {
        200: {
          body: _annotations_schema.annotationsResponseSchema,
          description: 'Get annotations response'
        }
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    request,
    response
  }) => {
    try {
      const {
        getAnnotations
      } = (0, _annotation_service.annotationServiceProvider)(client);
      const resp = await getAnnotations(request.body);
      return response.ok({
        body: resp
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));

  /**
   * @apiGroup Annotations
   */
  router.versioned.put({
    path: `${_app.ML_INTERNAL_BASE_PATH}/annotations/index`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canCreateAnnotation']
      }
    },
    summary: 'Indexes annotation',
    description: 'Indexes the annotation.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: _annotations_schema.indexAnnotationSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    request,
    response
  }) => {
    try {
      var _currentUser$username;
      const annotationsFeatureAvailable = await (0, _check_annotations.isAnnotationsFeatureAvailable)(client);
      if (annotationsFeatureAvailable === false) {
        throw getAnnotationsFeatureUnavailableErrorMessage();
      }
      const {
        indexAnnotation
      } = (0, _annotation_service.annotationServiceProvider)(client);
      const currentUser = securityPlugin !== undefined ? securityPlugin.authc.getCurrentUser(request) : {};
      // @ts-expect-error username doesn't exist on {}
      const username = (_currentUser$username = currentUser === null || currentUser === void 0 ? void 0 : currentUser.username) !== null && _currentUser$username !== void 0 ? _currentUser$username : _annotations.ANNOTATION_USER_UNKNOWN;
      const resp = await indexAnnotation(request.body, username);
      return response.ok({
        body: resp
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));

  /**
   * @apiGroup Annotations
   */
  router.versioned.delete({
    path: `${_app.ML_INTERNAL_BASE_PATH}/annotations/delete/{annotationId}`,
    access: 'internal',
    security: {
      authz: {
        requiredPrivileges: ['ml:canDeleteAnnotation']
      }
    },
    summary: 'Deletes annotation',
    description: 'Deletes the specified annotation.'
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: _annotations_schema.deleteAnnotationSchema
      }
    }
  }, routeGuard.fullLicenseAPIGuard(async ({
    client,
    request,
    response
  }) => {
    try {
      const annotationsFeatureAvailable = await (0, _check_annotations.isAnnotationsFeatureAvailable)(client);
      if (annotationsFeatureAvailable === false) {
        throw getAnnotationsFeatureUnavailableErrorMessage();
      }
      const annotationId = request.params.annotationId;
      const {
        deleteAnnotation
      } = (0, _annotation_service.annotationServiceProvider)(client);
      const resp = await deleteAnnotation(annotationId);
      return response.ok({
        body: resp
      });
    } catch (e) {
      return response.customError((0, _error_wrapper.wrapError)(e));
    }
  }));
}