"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateTextWidth = calculateTextWidth;
exports.detectorToString = detectorToString;
exports.escapeForElasticsearchQuery = escapeForElasticsearchQuery;
exports.escapeKueryForEmbeddableFieldValuePair = escapeKueryForEmbeddableFieldValuePair;
exports.escapeKueryForFieldValuePair = escapeKueryForFieldValuePair;
exports.mlEscape = mlEscape;
exports.replaceStringTokens = replaceStringTokens;
exports.stringMatch = stringMatch;
exports.toLocaleString = toLocaleString;
var _d = _interopRequireDefault(require("d3"));
var _he = _interopRequireDefault(require("he"));
var _esQuery = require("@kbn/es-query");
var _mlIsDefined = require("@kbn/ml-is-defined");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
 * Contains utility functions for performing operations on Strings.
 */

// Replaces all instances of dollar delimited tokens in the specified String
// with corresponding values from the supplied object, optionally
// encoding the replacement for a URI component.
// For example if passed a String 'http://www.google.co.uk/#q=airline+code+$airline$'
// and valuesByTokenName of {"airline":"AAL"}, will return
// 'http://www.google.co.uk/#q=airline+code+AAL'.
// If a corresponding key is not found in valuesByTokenName, then the String is not replaced.
function replaceStringTokens(str, valuesByTokenName, encodeForURI) {
  return String(str).replace(/\$([^?&$\'"]+)\$/g, (match, name) => {
    // Use lodash get to allow nested JSON fields to be retrieved.
    let tokenValue = valuesByTokenName && valuesByTokenName[name] !== undefined ? valuesByTokenName[name] : null;
    if (encodeForURI === true && tokenValue !== null) {
      tokenValue = encodeURIComponent(tokenValue);
    }

    // If property not found string is not replaced.
    return tokenValue !== null ? tokenValue : match;
  });
}

// creates the default description for a given detector
function detectorToString(dtr) {
  const BY_TOKEN = ' by ';
  const OVER_TOKEN = ' over ';
  const USE_NULL_OPTION = ' use_null=';
  const PARTITION_FIELD_OPTION = ' partition_field_name=';
  const EXCLUDE_FREQUENT_OPTION = ' exclude_frequent=';
  let txt = '';
  if (dtr.function !== undefined && dtr.function !== '') {
    txt += dtr.function;
    if (dtr.field_name !== undefined && dtr.field_name !== '') {
      txt += '(' + quoteField(dtr.field_name) + ')';
    }
  } else if (dtr.field_name !== undefined && dtr.field_name !== '') {
    txt += quoteField(dtr.field_name);
  }
  if (dtr.by_field_name !== undefined && dtr.by_field_name !== '') {
    txt += BY_TOKEN + quoteField(dtr.by_field_name);
  }
  if (dtr.over_field_name !== undefined && dtr.over_field_name !== '') {
    txt += OVER_TOKEN + quoteField(dtr.over_field_name);
  }
  if (dtr.use_null !== undefined) {
    txt += USE_NULL_OPTION + dtr.use_null;
  }
  if (dtr.partition_field_name !== undefined && dtr.partition_field_name !== '') {
    txt += PARTITION_FIELD_OPTION + quoteField(dtr.partition_field_name);
  }
  if (dtr.exclude_frequent !== undefined) {
    txt += EXCLUDE_FREQUENT_OPTION + dtr.exclude_frequent;
  }
  return txt;
}

// wrap a the inputed string in quotes if it contains non-word characters
function quoteField(field) {
  if (field.match(/\W/g)) {
    return '"' + field + '"';
  } else {
    return field;
  }
}

// add commas to large numbers
// Number.toLocaleString is not supported on safari
function toLocaleString(x) {
  if (x === undefined || x === null) {
    return '';
  }
  let result = x.toString();
  if (x && typeof x === 'number') {
    const parts = x.toString().split('.');
    parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, ',');
    result = parts.join('.');
  }
  return result;
}

// escape html characters
function mlEscape(str) {
  // It's not possible to use "he" encoding directly
  // because \ and / characters are not going to be replaced without
  // encodeEverything option. But with this option enabled
  // each word character is encoded as well.
  return String(str).replace(/\W/g, s => _he.default.encode(s, {
    useNamedReferences: true,
    encodeEverything: true,
    allowUnsafeSymbols: false
  }));
}

// Escapes reserved characters for use in Elasticsearch query terms.
function escapeForElasticsearchQuery(str) {
  // Escape with a leading backslash any of the characters that
  // Elastic document may cause a syntax error when used in queries:
  // + - = && || > < ! ( ) { } [ ] ^ " ~ * ? : \ /
  // https://www.elastic.co/guide/en/elasticsearch/reference/current/query-dsl-query-string-query.html#_reserved_characters
  return String(str).replace(/[-[\]{}()+!<>=?:\/\\^"~*&|\s]/g, '\\$&');
}
function escapeKueryForFieldValuePair(name, value) {
  if (!(0, _mlIsDefined.isDefined)(name) || !(0, _mlIsDefined.isDefined)(value)) return '';
  return `${(0, _esQuery.escapeKuery)(name)}:${(0, _esQuery.escapeKuery)(value.toString())}`;
}
const replaceEmptyStringWithQuotation = s => s === '' ? '""' : s;

/**
 *
 * Helper function to returns escaped combined field name and value
 * which also replaces empty str with " to ensure compatability with kql queries
 * @param name fieldName of selection
 * @param value fieldValue of selection
 * @returns {string} escaped `name:value` compatible with embeddable input
 */
function escapeKueryForEmbeddableFieldValuePair(name, value) {
  if (!(0, _mlIsDefined.isDefined)(name) || !(0, _mlIsDefined.isDefined)(value)) return '';
  return `${replaceEmptyStringWithQuotation((0, _esQuery.escapeKuery)(name))}:${replaceEmptyStringWithQuotation((0, _esQuery.escapeKuery)(value.toString()))}`;
}
function calculateTextWidth(txt, isNumber) {
  txt = isNumber && typeof txt === 'number' ? _d.default.format(',')(txt) : txt;

  // Create a temporary selection to append the label to.
  // Note styling of font will be inherited from CSS of page.
  const $body = _d.default.select('body');
  const $el = $body.append('div');
  const svg = $el.append('svg');
  const tempLabelText = svg.append('g').attr('class', 'temp-axis-label tick').selectAll('text.temp.axis').data(['a']).enter().append('text').text(txt);
  const width = tempLabelText[0][0].getBBox().width;
  _d.default.select('.temp-axis-label').remove();
  if ($el !== undefined) {
    $el.remove();
  }
  return Math.ceil(width);
}
function stringMatch(str, substr) {
  try {
    return typeof str === 'string' && typeof substr === 'string' && str.toLowerCase().match(substr.toLowerCase()) === null === false;
  } catch (error) {
    return false;
  }
}