"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useDeleteAnalyticsAndDestIndex = exports.useDeleteAnalytics = exports.useCanDeleteIndex = void 0;
var _i18n = require("@kbn/i18n");
var _mlErrorUtils = require("@kbn/ml-error-utils");
var _kibana = require("../../../../../contexts/kibana");
var _toast_notification_service = require("../../../../../services/toast_notification_service");
var _common = require("../../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useDeleteAnalytics = () => {
  const toastNotificationService = (0, _toast_notification_service.useToastNotificationService)();
  const mlApi = (0, _kibana.useMlApi)();
  return async analyticsConfig => {
    try {
      await mlApi.dataFrameAnalytics.deleteDataFrameAnalytics(analyticsConfig.id);
      toastNotificationService.displaySuccessToast(_i18n.i18n.translate('xpack.ml.dataframe.analyticsList.deleteAnalyticsSuccessMessage', {
        defaultMessage: 'Request to delete data frame analytics job {analyticsId} acknowledged.',
        values: {
          analyticsId: analyticsConfig.id
        }
      }));
    } catch (e) {
      toastNotificationService.displayErrorToast(e, _i18n.i18n.translate('xpack.ml.dataframe.analyticsList.deleteAnalyticsErrorMessage', {
        defaultMessage: 'An error occurred deleting the data frame analytics job {analyticsId}',
        values: {
          analyticsId: analyticsConfig.id
        }
      }));
    }
    _common.refreshAnalyticsList$.next(_common.REFRESH_ANALYTICS_LIST_STATE.REFRESH);
  };
};
exports.useDeleteAnalytics = useDeleteAnalytics;
const useDeleteAnalyticsAndDestIndex = () => {
  const toastNotificationService = (0, _toast_notification_service.useToastNotificationService)();
  const mlApi = (0, _kibana.useMlApi)();
  return async (analyticsConfig, deleteDestIndex, deleteDestDataView) => {
    const destinationIndex = analyticsConfig.dest.index;
    try {
      var _status$analyticsJobD, _status$analyticsJobD2, _status$destIndexDele, _status$destIndexDele2, _status$destDataViewD, _status$destDataViewD2;
      const status = await mlApi.dataFrameAnalytics.deleteDataFrameAnalyticsAndDestIndex(analyticsConfig.id, deleteDestIndex, deleteDestDataView);
      if ((_status$analyticsJobD = status.analyticsJobDeleted) !== null && _status$analyticsJobD !== void 0 && _status$analyticsJobD.success) {
        toastNotificationService.displaySuccessToast(_i18n.i18n.translate('xpack.ml.dataframe.analyticsList.deleteAnalyticsSuccessMessage', {
          defaultMessage: 'Request to delete data frame analytics job {analyticsId} acknowledged.',
          values: {
            analyticsId: analyticsConfig.id
          }
        }));
      }
      if ((_status$analyticsJobD2 = status.analyticsJobDeleted) !== null && _status$analyticsJobD2 !== void 0 && _status$analyticsJobD2.error) {
        toastNotificationService.displayErrorToast(status.analyticsJobDeleted.error, _i18n.i18n.translate('xpack.ml.dataframe.analyticsList.deleteAnalyticsErrorMessage', {
          defaultMessage: 'An error occurred deleting the data frame analytics job {analyticsId}',
          values: {
            analyticsId: analyticsConfig.id
          }
        }));
      }
      if ((_status$destIndexDele = status.destIndexDeleted) !== null && _status$destIndexDele !== void 0 && _status$destIndexDele.success) {
        toastNotificationService.displaySuccessToast(_i18n.i18n.translate('xpack.ml.dataframe.analyticsList.deleteAnalyticsWithIndexSuccessMessage', {
          defaultMessage: 'Request to delete destination index {destinationIndex} acknowledged.',
          values: {
            destinationIndex
          }
        }));
      }
      if ((_status$destIndexDele2 = status.destIndexDeleted) !== null && _status$destIndexDele2 !== void 0 && _status$destIndexDele2.error) {
        toastNotificationService.displayErrorToast(status.destIndexDeleted.error, _i18n.i18n.translate('xpack.ml.dataframe.analyticsList.deleteAnalyticsWithIndexErrorMessage', {
          defaultMessage: 'An error occurred deleting destination index {destinationIndex}',
          values: {
            destinationIndex
          }
        }));
      }
      if ((_status$destDataViewD = status.destDataViewDeleted) !== null && _status$destDataViewD !== void 0 && _status$destDataViewD.success) {
        toastNotificationService.displaySuccessToast(_i18n.i18n.translate('xpack.ml.dataframe.analyticsList.deleteAnalyticsWithDataViewSuccessMessage', {
          defaultMessage: 'Request to delete data view {destinationIndex} acknowledged.',
          values: {
            destinationIndex
          }
        }));
      }
      if ((_status$destDataViewD2 = status.destDataViewDeleted) !== null && _status$destDataViewD2 !== void 0 && _status$destDataViewD2.error) {
        const error = (0, _mlErrorUtils.extractErrorMessage)(status.destDataViewDeleted.error);
        toastNotificationService.displayDangerToast(_i18n.i18n.translate('xpack.ml.dataframe.analyticsList.deleteAnalyticsWithDataViewErrorMessage', {
          defaultMessage: 'An error occurred deleting data view {destinationIndex}: {error}',
          values: {
            destinationIndex,
            error
          }
        }));
      }
    } catch (e) {
      toastNotificationService.displayErrorToast(e, _i18n.i18n.translate('xpack.ml.dataframe.analyticsList.deleteAnalyticsErrorMessage', {
        defaultMessage: 'An error occurred deleting the data frame analytics job {analyticsId}',
        values: {
          analyticsId: analyticsConfig.id
        }
      }));
    }
    _common.refreshAnalyticsList$.next(_common.REFRESH_ANALYTICS_LIST_STATE.REFRESH);
  };
};
exports.useDeleteAnalyticsAndDestIndex = useDeleteAnalyticsAndDestIndex;
const useCanDeleteIndex = () => {
  const toastNotificationService = (0, _toast_notification_service.useToastNotificationService)();
  const mlApi = (0, _kibana.useMlApi)();
  return async indexName => {
    try {
      const privilege = await mlApi.hasPrivileges({
        index: [{
          names: [indexName],
          // uses wildcard
          privileges: ['delete_index']
        }]
      });
      if (!privilege) {
        return false;
      }
      return privilege.hasPrivileges === undefined || privilege.hasPrivileges.has_all_requested === true;
    } catch (e) {
      const error = (0, _mlErrorUtils.extractErrorMessage)(e);
      toastNotificationService.displayDangerToast(_i18n.i18n.translate('xpack.ml.dataframe.analyticsList.deleteAnalyticsPrivilegeErrorMessage', {
        defaultMessage: 'User does not have permission to delete index {indexName}: {error}',
        values: {
          indexName,
          error
        }
      }));
    }
  };
};
exports.useCanDeleteIndex = useCanDeleteIndex;