"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.metricByField = void 0;
exports.useHostMetricsTable = useHostMetricsTable;
var _react = require("react");
var _shared = require("../shared");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const hostsMetricsQueryConfig = {
  sourceFilter: {
    term: {
      'event.module': 'system'
    }
  },
  groupByField: 'host.name',
  metricsMap: {
    'system.cpu.cores': {
      aggregation: 'max',
      field: 'system.cpu.cores'
    },
    'system.cpu.total.norm.pct': {
      aggregation: 'avg',
      field: 'system.cpu.total.norm.pct'
    },
    'system.memory.total': {
      aggregation: 'max',
      field: 'system.memory.total'
    },
    'system.memory.used.pct': {
      aggregation: 'avg',
      field: 'system.memory.used.pct'
    }
  }
};
const metricByField = exports.metricByField = (0, _shared.createMetricByFieldLookup)(hostsMetricsQueryConfig.metricsMap);
const unpackMetric = (0, _shared.makeUnpackMetric)(metricByField);
function useHostMetricsTable({
  timerange,
  filterClauseDsl,
  metricsClient
}) {
  const [currentPageIndex, setCurrentPageIndex] = (0, _react.useState)(0);
  const [sortState, setSortState] = (0, _react.useState)({
    field: 'averageCpuUsagePercent',
    direction: 'desc'
  });
  const {
    options: hostMetricsOptions
  } = (0, _react.useMemo)(() => (0, _shared.metricsToApiOptions)(hostsMetricsQueryConfig, filterClauseDsl), [filterClauseDsl]);
  const {
    data,
    isLoading
  } = (0, _shared.useInfrastructureNodeMetrics)({
    metricsExplorerOptions: hostMetricsOptions,
    timerange,
    transform: seriesToHostNodeMetricsRow,
    sortState,
    currentPageIndex,
    metricsClient
  });
  return {
    data,
    isLoading,
    setCurrentPageIndex,
    setSortState,
    sortState,
    timerange
  };
}
function seriesToHostNodeMetricsRow(series) {
  if (series.rows.length === 0) {
    return rowWithoutMetrics(series.id);
  }
  return {
    name: series.id,
    ...calculateMetricAverages(series.rows)
  };
}
function rowWithoutMetrics(name) {
  return {
    name,
    cpuCount: null,
    averageCpuUsagePercent: null,
    totalMemoryMegabytes: null,
    averageMemoryUsagePercent: null
  };
}
function calculateMetricAverages(rows) {
  const {
    cpuCountValues,
    averageCpuUsagePercentValues,
    totalMemoryMegabytesValues,
    averageMemoryUsagePercentValues
  } = collectMetricValues(rows);
  let cpuCount = null;
  if (cpuCountValues.length !== 0) {
    cpuCount = cpuCountValues.at(-1);
  }
  let averageCpuUsagePercent = null;
  if (averageCpuUsagePercentValues.length !== 0) {
    averageCpuUsagePercent = (0, _shared.scaleUpPercentage)((0, _shared.averageOfValues)(averageCpuUsagePercentValues));
  }
  let totalMemoryMegabytes = null;
  if (totalMemoryMegabytesValues.length !== 0) {
    const memoryInBytes = totalMemoryMegabytesValues.at(-1);
    const bytesPerMegabyte = 1000000;
    totalMemoryMegabytes = Math.floor(memoryInBytes / bytesPerMegabyte);
  }
  let averageMemoryUsagePercent = null;
  if (averageMemoryUsagePercentValues.length !== 0) {
    averageMemoryUsagePercent = (0, _shared.scaleUpPercentage)((0, _shared.averageOfValues)(averageMemoryUsagePercentValues));
  }
  return {
    cpuCount,
    averageCpuUsagePercent,
    totalMemoryMegabytes,
    averageMemoryUsagePercent
  };
}
function collectMetricValues(rows) {
  const cpuCountValues = [];
  const averageCpuUsagePercentValues = [];
  const totalMemoryMegabytesValues = [];
  const averageMemoryUsagePercentValues = [];
  rows.forEach(row => {
    const {
      cpuCount,
      averageCpuUsagePercent,
      totalMemoryMegabytes,
      averageMemoryUsagePercent
    } = unpackMetrics(row);
    if (cpuCount !== null) {
      cpuCountValues.push(cpuCount);
    }
    if (averageCpuUsagePercent !== null) {
      averageCpuUsagePercentValues.push(averageCpuUsagePercent);
    }
    if (totalMemoryMegabytes !== null) {
      totalMemoryMegabytesValues.push(totalMemoryMegabytes);
    }
    if (averageMemoryUsagePercent !== null) {
      averageMemoryUsagePercentValues.push(averageMemoryUsagePercent);
    }
  });
  return {
    cpuCountValues,
    averageCpuUsagePercentValues,
    totalMemoryMegabytesValues,
    averageMemoryUsagePercentValues
  };
}
function unpackMetrics(row) {
  return {
    cpuCount: unpackMetric(row, 'system.cpu.cores'),
    averageCpuUsagePercent: unpackMetric(row, 'system.cpu.total.norm.pct'),
    totalMemoryMegabytes: unpackMetric(row, 'system.memory.total'),
    averageMemoryUsagePercent: unpackMetric(row, 'system.memory.used.pct')
  };
}