"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LOCKS_INDEX_TEMPLATE_NAME = exports.LOCKS_CONCRETE_INDEX_NAME = exports.LOCKS_COMPONENT_TEMPLATE_NAME = void 0;
exports.ensureTemplatesAndIndexCreated = ensureTemplatesAndIndexCreated;
exports.removeLockIndexWithIncorrectMappings = removeLockIndexWithIncorrectMappings;
exports.setupLockManagerIndex = setupLockManagerIndex;
var _elasticsearch = require("@elastic/elasticsearch");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const LOCKS_INDEX_ALIAS = '.kibana_locks';
const INDEX_PATTERN = `${LOCKS_INDEX_ALIAS}*`;
const LOCKS_CONCRETE_INDEX_NAME = exports.LOCKS_CONCRETE_INDEX_NAME = `${LOCKS_INDEX_ALIAS}-000001`;
const LOCKS_COMPONENT_TEMPLATE_NAME = exports.LOCKS_COMPONENT_TEMPLATE_NAME = `${LOCKS_INDEX_ALIAS}-component`;
const LOCKS_INDEX_TEMPLATE_NAME = exports.LOCKS_INDEX_TEMPLATE_NAME = `${LOCKS_INDEX_ALIAS}-index-template`;
async function removeLockIndexWithIncorrectMappings(esClient, logger) {
  var _mappings$properties, _mappings$properties$, _mappings$properties2, _mappings$properties3;
  let res;
  try {
    res = await esClient.indices.getMapping({
      index: LOCKS_CONCRETE_INDEX_NAME
    });
  } catch (error) {
    const isNotFoundError = error instanceof _elasticsearch.errors.ResponseError && error.statusCode === 404;
    if (!isNotFoundError) {
      logger.error(`Failed to get mapping for lock index "${LOCKS_CONCRETE_INDEX_NAME}": ${error.message}`);
    }
    return;
  }
  const indexMappings = Object.values(res);
  if (indexMappings.length === 0) {
    logger.warn(`No mappings found for "${LOCKS_CONCRETE_INDEX_NAME}"`);
    return;
  }
  const {
    mappings
  } = indexMappings[0];
  if (!mappings) {
    logger.debug(`Mappings object is empty for "${LOCKS_CONCRETE_INDEX_NAME}"`);
    return;
  }
  const hasIncorrectMappings = ((_mappings$properties = mappings.properties) === null || _mappings$properties === void 0 ? void 0 : (_mappings$properties$ = _mappings$properties.token) === null || _mappings$properties$ === void 0 ? void 0 : _mappings$properties$.type) !== 'keyword' || ((_mappings$properties2 = mappings.properties) === null || _mappings$properties2 === void 0 ? void 0 : (_mappings$properties3 = _mappings$properties2.expiresAt) === null || _mappings$properties3 === void 0 ? void 0 : _mappings$properties3.type) !== 'date';
  if (hasIncorrectMappings) {
    logger.warn(`Lock index "${LOCKS_CONCRETE_INDEX_NAME}" has incorrect mappings.`);
    try {
      await esClient.indices.delete({
        index: LOCKS_CONCRETE_INDEX_NAME
      });
      logger.info(`Lock index "${LOCKS_CONCRETE_INDEX_NAME}" removed successfully.`);
    } catch (error) {
      logger.error(`Failed to remove lock index "${LOCKS_CONCRETE_INDEX_NAME}": ${error.message}`);
    }
  }
}
async function ensureTemplatesAndIndexCreated(esClient, logger) {
  await esClient.cluster.putComponentTemplate({
    name: LOCKS_COMPONENT_TEMPLATE_NAME,
    template: {
      mappings: {
        dynamic: false,
        properties: {
          token: {
            type: 'keyword'
          },
          metadata: {
            enabled: false
          },
          createdAt: {
            type: 'date'
          },
          expiresAt: {
            type: 'date'
          }
        }
      }
    }
  });
  logger.info(`Component template ${LOCKS_COMPONENT_TEMPLATE_NAME} created or updated successfully.`);
  await esClient.indices.putIndexTemplate({
    name: LOCKS_INDEX_TEMPLATE_NAME,
    index_patterns: [INDEX_PATTERN],
    composed_of: [LOCKS_COMPONENT_TEMPLATE_NAME],
    priority: 500,
    template: {
      settings: {
        number_of_shards: 1,
        auto_expand_replicas: '0-1',
        hidden: true
      }
    }
  });
  logger.info(`Index template ${LOCKS_INDEX_TEMPLATE_NAME} created or updated successfully.`);
  try {
    await esClient.indices.create({
      index: LOCKS_CONCRETE_INDEX_NAME
    });
    logger.info(`Index ${LOCKS_CONCRETE_INDEX_NAME} created successfully.`);
  } catch (error) {
    const isIndexAlreadyExistsError = error instanceof _elasticsearch.errors.ResponseError && error.body.error.type === 'resource_already_exists_exception';

    // Handle the case where the index name already exists as an alias (e.g., after a reindex operation during cluster upgrade)
    const isIndexNameExistsAsAliasError = error instanceof _elasticsearch.errors.ResponseError && error.body.error.type === 'invalid_index_name_exception';
    if (isIndexAlreadyExistsError || isIndexNameExistsAsAliasError) {
      logger.debug(`Index ${LOCKS_CONCRETE_INDEX_NAME} already exists. Skipping creation.`);
      return;
    }
    logger.error(`Unable to create index ${LOCKS_CONCRETE_INDEX_NAME}: ${error.message}`);
    throw error;
  }
}
async function setupLockManagerIndex(esClient, logger) {
  await removeLockIndexWithIncorrectMappings(esClient, logger); // TODO: should be removed in the future (after 9.1). See https://github.com/elastic/kibana/issues/218944
  await ensureTemplatesAndIndexCreated(esClient, logger);
}