"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ExceptionListClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _securitysolutionIoTsListTypes = require("@kbn/securitysolution-io-ts-list-types");
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _utils = require("@kbn/utils");
var _get_exception_list = require("./get_exception_list");
var _export_exception_list_and_items = require("./export_exception_list_and_items");
var _get_exception_list_summary = require("./get_exception_list_summary");
var _create_exception_list = require("./create_exception_list");
var _get_exception_list_item = require("./get_exception_list_item");
var _create_exception_list_item = require("./create_exception_list_item");
var _update_exception_list = require("./update_exception_list");
var _update_exception_list_item = require("./update_exception_list_item");
var _delete_exception_list = require("./delete_exception_list");
var _delete_exception_list_item = require("./delete_exception_list_item");
var _find_exception_list_item = require("./find_exception_list_item");
var _find_exception_list = require("./find_exception_list");
var _find_exception_list_items = require("./find_exception_list_items");
var _create_endpoint_list = require("./create_endpoint_list");
var _create_endpoint_trusted_apps_list = require("./create_endpoint_trusted_apps_list");
var _import_exception_list_and_items = require("./import_exception_list_and_items");
var _utils2 = require("./utils");
var _create_exceptions_stream_logic = require("./utils/import/create_exceptions_stream_logic");
var _open_point_in_time = require("./open_point_in_time");
var _close_point_in_time = require("./close_point_in_time");
var _find_exception_list_point_in_time_finder = require("./find_exception_list_point_in_time_finder");
var _find_value_list_exception_list_items = require("./find_value_list_exception_list_items");
var _find_exception_list_items_point_in_time_finder = require("./find_exception_list_items_point_in_time_finder");
var _find_value_list_exception_list_items_point_in_time_finder = require("./find_value_list_exception_list_items_point_in_time_finder");
var _find_exception_list_item_point_in_time_finder = require("./find_exception_list_item_point_in_time_finder");
var _duplicate_exception_list = require("./duplicate_exception_list");
var _update_overwrite_exception_list_item = require("./update_overwrite_exception_list_item");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Class for use for exceptions that are with trusted applications or
 * with rules.
 */
class ExceptionListClient {
  /**
   * Constructs the exception list
   * @param options
   * @param options.user The user associated with the action for exception list
   * @param options.savedObjectsClient The saved objects client to create, modify, delete, an exception list
   * @param options.serverExtensionsClient The server extensions client that can be useful for injecting domain specific rules
   * @param options.request optionally, the Kibana request which is useful for extension points
   */
  constructor({
    user: _user,
    savedObjectsClient: _savedObjectsClient,
    serverExtensionsClient,
    enableServerExtensionPoints = true,
    request
  }) {
    /** User creating, modifying, deleting, or updating an exception list */
    (0, _defineProperty2.default)(this, "user", void 0);
    /** Saved objects client to create, modify, delete, an exception list */
    (0, _defineProperty2.default)(this, "savedObjectsClient", void 0);
    /** server extensions client that can be useful for injecting domain specific rules */
    (0, _defineProperty2.default)(this, "serverExtensionsClient", void 0);
    /** Set to true to enable the server extension points, otherwise false */
    (0, _defineProperty2.default)(this, "enableServerExtensionPoints", void 0);
    /** Optionally, the Kibana request which is useful for extension points */
    (0, _defineProperty2.default)(this, "request", void 0);
    /**
     * Fetch an exception list parent container
     * @param options
     * @param options.listId the "list_id" of an exception list
     * @param options.id the "id" of an exception list
     * @param options.namespaceType saved object namespace (single | agnostic)
     * @returns The found exception list or null if none exists
     */
    (0, _defineProperty2.default)(this, "getExceptionList", async ({
      listId,
      id,
      namespaceType
    }) => {
      const {
        savedObjectsClient
      } = this;
      return (0, _get_exception_list.getExceptionList)({
        id,
        listId,
        namespaceType,
        savedObjectsClient
      });
    });
    /**
     * Fetch an exception list parent container
     * @param options.filter kql "filter" expression
     * @param options.listId the "list_id" of an exception list
     * @param options.id the "id" of an exception list
     * @param options.namespaceType saved object namespace (single | agnostic)
     *
     * @returns Summary of exception list item os types
     */
    (0, _defineProperty2.default)(this, "getExceptionListSummary", async options => {
      const {
        savedObjectsClient
      } = this;
      if (this.enableServerExtensionPoints) {
        await this.serverExtensionsClient.pipeRun('exceptionsListPreSummary', options, this.getServerExtensionCallbackContext());
      }
      return (0, _get_exception_list_summary.getExceptionListSummary)({
        ...options,
        savedObjectsClient
      });
    });
    /**
     * Fetch an exception list item container
     * @param options
     * @param options.itemId the "item_id" of an exception list (Either this or id has to be defined)
     * @param options.id the "id" of an exception list (Either this or itemId has to be defined)
     * @param options.namespaceType saved object namespace (single | agnostic)
     * @returns the found exception list item or null if none exists
     */
    (0, _defineProperty2.default)(this, "getExceptionListItem", async ({
      itemId,
      id,
      namespaceType
    }) => {
      const {
        savedObjectsClient
      } = this;
      if (this.enableServerExtensionPoints) {
        await this.serverExtensionsClient.pipeRun('exceptionsListPreGetOneItem', {
          id,
          itemId,
          namespaceType
        }, this.getServerExtensionCallbackContext());
      }
      return (0, _get_exception_list_item.getExceptionListItem)({
        id,
        itemId,
        namespaceType,
        savedObjectsClient
      });
    });
    /**
     * This creates an agnostic space endpoint list if it does not exist. This tries to be
     * as fast as possible by ignoring conflict errors and not returning the contents of the
     * list if it already exists.
     * @returns ExceptionListSchema if it created the endpoint list, otherwise null if it already exists
     */
    (0, _defineProperty2.default)(this, "createEndpointList", async () => {
      const {
        savedObjectsClient,
        user
      } = this;
      return (0, _create_endpoint_list.createEndpointList)({
        savedObjectsClient,
        user,
        version: 1
      });
    });
    /**
     * Create the Trusted Apps Agnostic list if it does not yet exist (`null` is returned if it does exist)
     * @returns The exception list schema or null if it does not exist
     */
    (0, _defineProperty2.default)(this, "createTrustedAppsList", async () => {
      const {
        savedObjectsClient,
        user
      } = this;
      return (0, _create_endpoint_trusted_apps_list.createEndpointTrustedAppsList)({
        savedObjectsClient,
        user,
        version: 1
      });
    });
    /**
     * This is the same as "createListItem" except it applies specifically to the agnostic endpoint list and will
     * auto-call the "createEndpointList" for you so that you have the best chance of the agnostic endpoint
     * being there and existing before the item is inserted into the agnostic endpoint list.
     * @param options
     * @param options.comments The comments of the endpoint list item
     * @param options.description The description of the endpoint list item
     * @param options.entries The entries of the endpoint list item
     * @param options.itemId The item id of the list item
     * @param options.meta Optional meta data of the list item
     * @param options.name The name of the list item
     * @param options.osTypes The OS type of the list item
     * @param options.tags Tags of the endpoint list item
     * @param options.type The type of the endpoint list item (Default is "simple")
     * @returns The exception list item created, otherwise null if not created
     */
    (0, _defineProperty2.default)(this, "createEndpointListItem", async ({
      comments,
      description,
      entries,
      itemId,
      meta,
      name,
      osTypes,
      tags,
      type
    }) => {
      const {
        savedObjectsClient,
        user
      } = this;
      await this.createEndpointList();
      let itemData = {
        comments,
        description,
        entries,
        expireTime: undefined,
        // Not currently used with endpoint exceptions
        itemId,
        listId: _securitysolutionListConstants.ENDPOINT_LIST_ID,
        meta,
        name,
        namespaceType: 'agnostic',
        osTypes,
        tags,
        type
      };
      if (this.enableServerExtensionPoints) {
        itemData = await this.serverExtensionsClient.pipeRun('exceptionsListPreCreateItem', itemData, this.getServerExtensionCallbackContext(), data => {
          return (0, _utils2.validateData)(_securitysolutionIoTsListTypes.createExceptionListItemSchema, (0, _utils2.transformCreateExceptionListItemOptionsToCreateExceptionListItemSchema)(data));
        });
      }
      return (0, _create_exception_list_item.createExceptionListItem)({
        ...itemData,
        savedObjectsClient,
        user
      });
    });
    /**
     * Create the Trusted Apps Agnostic list if it does not yet exist (`null` is returned if it does exist)
     * @param options.list the "list" to be duplicated
     * @param options.namespaceType saved object namespace (single | agnostic)
     * @param options.includeExpiredExceptions include or exclude expired TTL exception items
     * @returns The exception list schema or null if it does not exist
     */
    (0, _defineProperty2.default)(this, "duplicateExceptionListAndItems", async ({
      list,
      namespaceType,
      includeExpiredExceptions
    }) => {
      const {
        savedObjectsClient,
        user
      } = this;
      return (0, _duplicate_exception_list.duplicateExceptionListAndItems)({
        includeExpiredExceptions,
        list,
        namespaceType,
        savedObjectsClient,
        user
      });
    });
    /**
     * This is the same as "updateExceptionListItem" except it applies specifically to the endpoint list and will
     * auto-call the "createEndpointList" for you so that you have the best chance of the endpoint
     * being there if it did not exist before. If the list did not exist before, then creating it here will still cause a
     * return of null but at least the list exists again.
     * @param options
     * @param options._version The version to update the endpoint list item to
     * @param options.comments The comments of the endpoint list item
     * @param options.description The description of the endpoint list item
     * @param options.entries The entries of the endpoint list item
     * @param options.id The id of the list item (Either this or itemId has to be defined)
     * @param options.itemId The item id of the list item (Either this or id has to be defined)
     * @param options.meta Optional meta data of the list item
     * @param options.name The name of the list item
     * @param options.osTypes The OS type of the list item
     * @param options.tags Tags of the endpoint list item
     * @param options.type The type of the endpoint list item (Default is "simple")
     * @returns The exception list item updated, otherwise null if not updated
     */
    (0, _defineProperty2.default)(this, "updateEndpointListItem", async ({
      _version,
      comments,
      description,
      entries,
      id,
      itemId,
      meta,
      name,
      osTypes,
      tags,
      type
    }) => {
      const {
        savedObjectsClient,
        user
      } = this;
      await this.createEndpointList();
      let updatedItem = {
        _version,
        comments,
        description,
        entries,
        expireTime: undefined,
        // Not currently used with endpoint exceptions
        id,
        itemId,
        meta,
        name,
        namespaceType: 'agnostic',
        osTypes,
        tags,
        type
      };
      if (this.enableServerExtensionPoints) {
        updatedItem = await this.serverExtensionsClient.pipeRun('exceptionsListPreUpdateItem', updatedItem, this.getServerExtensionCallbackContext(), data => {
          return (0, _utils2.validateData)(_securitysolutionIoTsListTypes.updateExceptionListItemSchema, (0, _utils2.transformUpdateExceptionListItemOptionsToUpdateExceptionListItemSchema)(data));
        });
      }
      return (0, _update_exception_list_item.updateExceptionListItem)({
        ...updatedItem,
        savedObjectsClient,
        user
      });
    });
    /**
     * This is the same as "getExceptionListItem" except it applies specifically to the endpoint list.
     * @param options
     * @param options.itemId The item id (Either this or id has to be defined)
     * @param options.id The id (Either this or itemId has to be defined)
     * @returns The exception list item found, otherwise null
     */
    (0, _defineProperty2.default)(this, "getEndpointListItem", async ({
      itemId,
      id
    }) => {
      const {
        savedObjectsClient
      } = this;
      if (this.enableServerExtensionPoints) {
        await this.serverExtensionsClient.pipeRun('exceptionsListPreGetOneItem', {
          id,
          itemId,
          namespaceType: 'agnostic'
        }, this.getServerExtensionCallbackContext());
      }
      return (0, _get_exception_list_item.getExceptionListItem)({
        id,
        itemId,
        namespaceType: 'agnostic',
        savedObjectsClient
      });
    });
    /**
     * Create an exception list container
     * @param options
     * @param options.description a description of the exception list
     * @param options.immutable True if it's a immutable list, otherwise false
     * @param options.listId the "list_id" of the exception list
     * @param options.meta Optional meta data to add to the exception list
     * @param options.name the "name" of the exception list
     * @param options.namespaceType saved object namespace (single | agnostic)
     * @param options.tags user assigned tags of exception list
     * @param options.type container type
     * @param options.version document version
     * @returns the created exception list parent container
     */
    (0, _defineProperty2.default)(this, "createExceptionList", async ({
      description,
      immutable,
      listId,
      meta,
      name,
      namespaceType,
      tags,
      type,
      version
    }) => {
      const {
        savedObjectsClient,
        user
      } = this;
      return (0, _create_exception_list.createExceptionList)({
        description,
        immutable,
        listId,
        meta,
        name,
        namespaceType,
        savedObjectsClient,
        tags,
        type,
        user,
        version
      });
    });
    /**
     * Update an existing exception list container
     * @param options
     * @param options._version document version
     * @param options.id the "id" of the exception list
     * @param options.description a description of the exception list
     * @param options.listId the "list_id" of the exception list
     * @param options.meta Optional meta object
     * @param options.name the "name" of the exception list
     * @param options.namespaceType saved object namespace (single | agnostic)
     * @param options.tags user assigned tags of exception list
     * @param options.type container type
     * @param options.version document version, if undefined the current version number will be auto-incremented
     * @returns the updated exception list parent container
     */
    (0, _defineProperty2.default)(this, "updateExceptionList", async ({
      _version,
      id,
      description,
      listId,
      meta,
      name,
      namespaceType,
      tags,
      type,
      version
    }) => {
      const {
        savedObjectsClient,
        user
      } = this;
      return (0, _update_exception_list.updateExceptionList)({
        _version,
        description,
        id,
        listId,
        meta,
        name,
        namespaceType,
        savedObjectsClient,
        tags,
        type,
        user,
        version
      });
    });
    /**
     * Delete an exception list container by either id or list_id
     * @param options
     * @param options.listId the "list_id" of an exception list (Either this or id has to be defined)
     * @param options.id the "id" of an exception list (Either this or listId has to be defined)
     * @param options.namespaceType saved object namespace (single | agnostic)
     * @returns the deleted exception list or null if none exists
     */
    (0, _defineProperty2.default)(this, "deleteExceptionList", async ({
      id,
      listId,
      namespaceType
    }) => {
      const {
        savedObjectsClient
      } = this;
      return (0, _delete_exception_list.deleteExceptionList)({
        id,
        listId,
        namespaceType,
        savedObjectsClient
      });
    });
    /**
     * Create an exception list item container
     * @param options
     * @param options.comments User comments for the exception list item
     * @param options.description a description of the exception list
     * @param options.entries an array with the exception list item entries
     * @param options.itemId the "item_id" of the exception list item
     * @param options.listId the "list_id" of the parent exception list
     * @param options.meta Optional meta data about the list item
     * @param options.name the "name" of the exception list
     * @param options.namespaceType saved object namespace (single | agnostic)
     * @param options.osTypes item os types to apply
     * @param options.tags user assigned tags of exception list
     * @param options.type container type
     * @returns the created exception list item container
     */
    (0, _defineProperty2.default)(this, "createExceptionListItem", async ({
      comments,
      description,
      entries,
      expireTime,
      itemId,
      listId,
      meta,
      name,
      namespaceType,
      osTypes,
      tags,
      type
    }) => {
      const {
        savedObjectsClient,
        user
      } = this;
      let itemData = {
        comments,
        description,
        entries,
        expireTime,
        itemId,
        listId,
        meta,
        name,
        namespaceType,
        osTypes,
        tags,
        type
      };
      if (this.enableServerExtensionPoints) {
        itemData = await this.serverExtensionsClient.pipeRun('exceptionsListPreCreateItem', itemData, this.getServerExtensionCallbackContext(), data => {
          return (0, _utils2.validateData)(_securitysolutionIoTsListTypes.createExceptionListItemSchema, (0, _utils2.transformCreateExceptionListItemOptionsToCreateExceptionListItemSchema)(data));
        });
      }
      return (0, _create_exception_list_item.createExceptionListItem)({
        ...itemData,
        savedObjectsClient,
        user
      });
    });
    /**
     * Update an existing exception list item
     *
     * NOTE: This method will PATCH the targeted exception list item, not fully overwrite it.
     * Any undefined fields passed in will not be changed in the existing record. To unset any
     * fields use the `updateOverwriteExceptionListItem` method
     *
     * @param options
     * @param options._version document version
     * @param options.comments user comments attached to item
     * @param options.entries item exception entries logic
     * @param options.id the "id" of the exception list item
     * @param options.description a description of the exception list
     * @param options.itemId the "item_id" of the exception list item
     * @param options.meta Optional meta data about the exception list item
     * @param options.name the "name" of the exception list
     * @param options.namespaceType saved object namespace (single | agnostic)
     * @param options.osTypes item os types to apply
     * @param options.tags user assigned tags of exception list
     * @param options.type container type
     * @returns the updated exception list item or null if none exists
     */
    (0, _defineProperty2.default)(this, "updateExceptionListItem", async ({
      _version,
      comments,
      description,
      entries,
      expireTime,
      id,
      itemId,
      meta,
      name,
      namespaceType,
      osTypes,
      tags,
      type
    }) => {
      const {
        savedObjectsClient,
        user
      } = this;
      let updatedItem = {
        _version,
        comments,
        description,
        entries,
        expireTime,
        id,
        itemId,
        meta,
        name,
        namespaceType,
        osTypes,
        tags,
        type
      };
      if (this.enableServerExtensionPoints) {
        updatedItem = await this.serverExtensionsClient.pipeRun('exceptionsListPreUpdateItem', updatedItem, this.getServerExtensionCallbackContext(), data => {
          return (0, _utils2.validateData)(_securitysolutionIoTsListTypes.updateExceptionListItemSchema, (0, _utils2.transformUpdateExceptionListItemOptionsToUpdateExceptionListItemSchema)(data));
        });
      }
      return (0, _update_exception_list_item.updateExceptionListItem)({
        ...updatedItem,
        savedObjectsClient,
        user
      });
    });
    /**
     * Update an existing exception list item using the overwrite method in order to behave
     * more like a PUT request rather than a PATCH request.
     *
     * This was done in order to correctly unset types via update which cannot be accomplished
     * using the regular `updateExceptionItem` method. All other results of the methods are identical
     *
     * @param options
     * @param options._version document version
     * @param options.comments user comments attached to item
     * @param options.entries item exception entries logic
     * @param options.id the "id" of the exception list item
     * @param options.description a description of the exception list
     * @param options.itemId the "item_id" of the exception list item
     * @param options.meta Optional meta data about the exception list item
     * @param options.name the "name" of the exception list
     * @param options.namespaceType saved object namespace (single | agnostic)
     * @param options.osTypes item os types to apply
     * @param options.tags user assigned tags of exception list
     * @param options.type container type
     * @returns the updated exception list item or null if none exists
     */
    (0, _defineProperty2.default)(this, "updateOverwriteExceptionListItem", async ({
      _version,
      comments,
      description,
      entries,
      expireTime,
      id,
      itemId,
      meta,
      name,
      namespaceType,
      osTypes,
      tags,
      type
    }) => {
      const {
        savedObjectsClient,
        user
      } = this;
      let updatedItem = {
        _version,
        comments,
        description,
        entries,
        expireTime,
        id,
        itemId,
        meta,
        name,
        namespaceType,
        osTypes,
        tags,
        type
      };
      if (this.enableServerExtensionPoints) {
        updatedItem = await this.serverExtensionsClient.pipeRun('exceptionsListPreUpdateItem', updatedItem, this.getServerExtensionCallbackContext(), data => {
          return (0, _utils2.validateData)(_securitysolutionIoTsListTypes.updateExceptionListItemSchema, (0, _utils2.transformUpdateExceptionListItemOptionsToUpdateExceptionListItemSchema)(data));
        });
      }
      return (0, _update_overwrite_exception_list_item.updateOverwriteExceptionListItem)({
        ...updatedItem,
        savedObjectsClient,
        user
      });
    });
    /**
     * Delete an exception list item by either id or item_id
     * @param options
     * @param options.itemId the "item_id" of an exception list item (Either this or id has to be defined)
     * @param options.id the "id" of an exception list item (Either this or itemId has to be defined)
     * @param options.namespaceType saved object namespace (single | agnostic)
     * @returns the deleted exception list item or null if none exists
     */
    (0, _defineProperty2.default)(this, "deleteExceptionListItem", async ({
      id,
      itemId,
      namespaceType
    }) => {
      const {
        savedObjectsClient
      } = this;
      if (this.enableServerExtensionPoints) {
        await this.serverExtensionsClient.pipeRun('exceptionsListPreDeleteItem', {
          id,
          itemId,
          namespaceType
        }, this.getServerExtensionCallbackContext());
      }
      return (0, _delete_exception_list_item.deleteExceptionListItem)({
        id,
        itemId,
        namespaceType,
        savedObjectsClient
      });
    });
    /**
     * Delete an exception list item by id
     * @param options
     * @param options.id the "id" of an exception list item
     * @param options.namespaceType saved object namespace (single | agnostic)
     */
    (0, _defineProperty2.default)(this, "deleteExceptionListItemById", async ({
      id,
      namespaceType
    }) => {
      const {
        savedObjectsClient
      } = this;
      if (this.enableServerExtensionPoints) {
        await this.serverExtensionsClient.pipeRun('exceptionsListPreDeleteItem', {
          id,
          itemId: undefined,
          namespaceType
        }, this.getServerExtensionCallbackContext());
      }
      return (0, _delete_exception_list_item.deleteExceptionListItemById)({
        id,
        namespaceType,
        savedObjectsClient
      });
    });
    /**
     * This is the same as "deleteExceptionListItem" except it applies specifically to the endpoint list.
     * Either id or itemId has to be defined to delete but not both is required. If both are provided, the id
     * is preferred.
     * @param options
     * @param options.id The id of the endpoint list item (Either this or itemId has to be defined)
     * @param options.itemId The item id of the endpoint list item (Either this or id has to be defined)
     */
    (0, _defineProperty2.default)(this, "deleteEndpointListItem", async ({
      id,
      itemId
    }) => {
      const {
        savedObjectsClient
      } = this;
      if (this.enableServerExtensionPoints) {
        await this.serverExtensionsClient.pipeRun('exceptionsListPreDeleteItem', {
          id,
          itemId,
          namespaceType: 'agnostic'
        }, this.getServerExtensionCallbackContext());
      }
      return (0, _delete_exception_list_item.deleteExceptionListItem)({
        id,
        itemId,
        namespaceType: 'agnostic',
        savedObjectsClient
      });
    });
    /**
     * Finds an exception list item given a set of criteria.
     * @param options
     * @param options.listId The single list id to do the search against
     * @param options.filter The filter to apply in the search
     * @param options.perPage How many per page to return
     * @param options.pit The Point in Time (pit) id if there is one, otherwise "undefined" can be send in
     * @param options.page The page number or "undefined" if there is no page number to continue from
     * @param options.searchAfter The search_after parameter if there is one, otherwise "undefined" can be sent in
     * @param options.sortField The sort field string if there is one, otherwise "undefined" can be sent in
     * @param options.sortOder The sort order string of "asc", "desc", otherwise "undefined" if there is no preference
     * @param options.namespaceType Set the list type of either "agnostic" | "single"
     * @returns The found exception list items or null if nothing is found
     */
    (0, _defineProperty2.default)(this, "findExceptionListItem", async ({
      listId,
      filter,
      perPage,
      pit,
      page,
      search,
      searchAfter,
      sortField,
      sortOrder,
      namespaceType
    }) => {
      const {
        savedObjectsClient
      } = this;
      const findOptions = {
        filter,
        listId,
        namespaceType,
        page,
        perPage,
        pit,
        searchAfter,
        sortField,
        sortOrder
      };
      if (this.enableServerExtensionPoints) {
        await this.serverExtensionsClient.pipeRun('exceptionsListPreSingleListFind', findOptions, this.getServerExtensionCallbackContext());
      }
      return (0, _find_exception_list_item.findExceptionListItem)({
        ...findOptions,
        savedObjectsClient,
        search
      });
    });
    /**
     * Finds exception lists items given a set of criteria.
     * @param options
     * @param options.listId The multiple list id's to do the search against
     * @param options.filter The filter to apply in the search
     * @param options.perPage How many per page to return
     * @param options.pit The Point in Time (pit) id if there is one, otherwise "undefined" can be sent in
     * @param options.page The page number or "undefined" if there is no page number to continue from
     * @param options.search The simple query search parameter if there is one, otherwise "undefined" can be sent in
     * @param options.searchAfter The search_after parameter if there is one, otherwise "undefined" can be sent in
     * @param options.sortField The sort field string if there is one, otherwise "undefined" can be sent in
     * @param options.sortOder The sort order string of "asc", "desc", otherwise "undefined" if there is no preference
     * @param options.namespaceType Set the list type of either "agnostic" | "single"
     * @returns The found exception lists items or null if nothing is found
     */
    (0, _defineProperty2.default)(this, "findExceptionListsItem", async ({
      listId,
      filter,
      perPage,
      pit,
      page,
      search,
      searchAfter,
      sortField,
      sortOrder,
      namespaceType
    }) => {
      const {
        savedObjectsClient
      } = this;
      if (this.enableServerExtensionPoints) {
        await this.serverExtensionsClient.pipeRun('exceptionsListPreMultiListFind', {
          filter,
          listId,
          namespaceType,
          page,
          perPage,
          pit,
          search,
          searchAfter,
          sortField,
          sortOrder
        }, this.getServerExtensionCallbackContext());
      }
      return (0, _find_exception_list_items.findExceptionListsItem)({
        filter,
        listId,
        namespaceType,
        page,
        perPage,
        pit,
        savedObjectsClient,
        search,
        searchAfter,
        sortField,
        sortOrder
      });
    });
    /**
     * Finds value list exception items given a set of criteria.
     * @param options
     * @param options.perPage How many per page to return
     * @param options.pit The Point in Time (pit) id if there is one, otherwise "undefined" can be send in
     * @param options.page The page number or "undefined" if there is no page number to continue from
     * @param options.searchAfter The search_after parameter if there is one, otherwise "undefined" can be sent in
     * @param options.sortField The sort field string if there is one, otherwise "undefined" can be sent in
     * @param options.sortOrder The sort order of "asc" or "desc", otherwise "undefined" can be sent in if there is no preference
     * @param options.valueListId The value list id
     * @returns The found value list exception list item or null if nothing is found
     */
    (0, _defineProperty2.default)(this, "findValueListExceptionListItems", async ({
      perPage,
      pit,
      page,
      searchAfter,
      sortField,
      sortOrder,
      valueListId
    }) => {
      const {
        savedObjectsClient
      } = this;
      return (0, _find_value_list_exception_list_items.findValueListExceptionListItems)({
        page,
        perPage,
        pit,
        savedObjectsClient,
        searchAfter,
        sortField,
        sortOrder,
        valueListId
      });
    });
    /**
     * Finds exception lists given a set of criteria.
     * @param options
     * @param options.filter The filter to apply in the search
     * @param options.perPage How many per page to return
     * @param options.page The page number or "undefined" if there is no page number to continue from
     * @param options.pit The Point in Time (pit) id if there is one, otherwise "undefined" can be sent in
     * @param options.searchAfter The search_after parameter if there is one, otherwise "undefined" can be sent in
     * @param options.sortField The sort field string if there is one, otherwise "undefined" can be sent in
     * @param options.sortOrder The sort order of "asc" or "desc", otherwise "undefined" can be sent in
     * @param options.namespaceType Set the list type of either "agnostic" | "single"
     * @returns The found exception lists or null if nothing is found
     */
    (0, _defineProperty2.default)(this, "findExceptionList", async ({
      filter,
      perPage,
      page,
      pit,
      searchAfter,
      sortField,
      sortOrder,
      namespaceType
    }) => {
      const {
        savedObjectsClient
      } = this;
      return (0, _find_exception_list.findExceptionList)({
        filter,
        namespaceType,
        page,
        perPage,
        pit,
        savedObjectsClient,
        searchAfter,
        sortField,
        sortOrder
      });
    });
    /**
     * This is the same as "findExceptionList" except it applies specifically to the  endpoint list and will
     * auto-call the "createEndpointList" for you so that you have the best chance of the  endpoint
     * being there if it did not exist before. If the list did not exist before, then creating it here should give you
     * a good guarantee that you will get an empty record set rather than null. I keep the null as the return value in
     * the off chance that you still might somehow not get into a race condition where the  endpoint list does
     * not exist because someone deleted it in-between the initial create and then the find.
     * @param options
     * @param options.filter The filter to apply in the search
     * @param options.perPage How many per page to return
     * @param options.page The page number or "undefined" if there is no page number to continue from
     * @param options.pit The Point in Time (pit) id if there is one, otherwise "undefined" can be sent in
     * @param options.search The simple query search parameter if there is one, otherwise "undefined" can be sent in
     * @param options.searchAfter The search_after parameter if there is one, otherwise "undefined" can be sent in
     * @param options.sortField The sort field string if there is one, otherwise "undefined" can be sent in
     * @param options.sortOrder The sort order of "asc" or "desc", otherwise "undefined" can be sent in
     * @returns The found exception list items or null if nothing is found
     */
    (0, _defineProperty2.default)(this, "findEndpointListItem", async ({
      filter,
      perPage,
      page,
      pit,
      search,
      searchAfter,
      sortField,
      sortOrder
    }) => {
      const {
        savedObjectsClient
      } = this;
      await this.createEndpointList();
      const findOptions = {
        filter,
        listId: _securitysolutionListConstants.ENDPOINT_LIST_ID,
        namespaceType: 'agnostic',
        page,
        perPage,
        pit,
        searchAfter,
        sortField,
        sortOrder
      };
      if (this.enableServerExtensionPoints) {
        await this.serverExtensionsClient.pipeRun('exceptionsListPreSingleListFind', findOptions, this.getServerExtensionCallbackContext());
      }
      return (0, _find_exception_list_item.findExceptionListItem)({
        ...findOptions,
        savedObjectsClient,
        search
      });
    });
    /**
     * Export an exception list parent container and it's items
     * @param options
     * @param options.listId the "list_id" of an exception list
     * @param options.id the "id" of an exception list
     * @param options.namespaceType saved object namespace (single | agnostic)
     * @param options.includeExpiredExceptions include or exclude expired TTL exception items
     * @returns the ndjson of the list and items to export or null if none exists
     */
    (0, _defineProperty2.default)(this, "exportExceptionListAndItems", async options => {
      const {
        savedObjectsClient
      } = this;
      if (this.enableServerExtensionPoints) {
        await this.serverExtensionsClient.pipeRun('exceptionsListPreExport', options, this.getServerExtensionCallbackContext());
      }
      return (0, _export_exception_list_and_items.exportExceptionListAndItems)({
        ...options,
        savedObjectsClient
      });
    });
    /**
     * Import exception lists parent containers and items as stream
     * @param options
     * @param options.exceptionsToImport ndjson stream of lists and items
     * @param options.maxExceptionsImportSize the max number of lists and items to import, defaults to 10,000
     * @param options.overwrite whether or not to overwrite an exception list with imported list if a matching list_id found
     * @returns summary of imported count and errors
     */
    (0, _defineProperty2.default)(this, "importExceptionListAndItems", async ({
      exceptionsToImport,
      maxExceptionsImportSize,
      overwrite,
      generateNewListId
    }) => {
      const {
        savedObjectsClient,
        user
      } = this;

      // validation of import and sorting of lists and items
      const readStream = (0, _create_exceptions_stream_logic.createExceptionsStreamFromNdjson)(maxExceptionsImportSize);
      const [parsedObjects] = await (0, _utils.createPromiseFromStreams)([exceptionsToImport, ...readStream]);
      if (this.enableServerExtensionPoints) {
        await this.serverExtensionsClient.pipeRun('exceptionsListPreImport', parsedObjects, this.getServerExtensionCallbackContext());
      }
      return (0, _import_exception_list_and_items.importExceptions)({
        exceptions: parsedObjects,
        generateNewListId,
        overwrite,
        savedObjectsClient,
        user
      });
    });
    /**
     * Import exception lists parent containers and items as array
     * @param options
     * @param options.exceptionsToImport array of lists and items
     * @param options.maxExceptionsImportSize the max number of lists and items to import, defaults to 10,000
     * @param options.overwrite whether or not to overwrite an exception list with imported list if a matching list_id found
     * @returns summary of imported count and errors
     */
    (0, _defineProperty2.default)(this, "importExceptionListAndItemsAsArray", async ({
      exceptionsToImport,
      maxExceptionsImportSize,
      overwrite
    }) => {
      const {
        savedObjectsClient,
        user
      } = this;

      // validation of import and sorting of lists and items
      const parsedObjects = (0, _create_exceptions_stream_logic.exceptionsChecksFromArray)(exceptionsToImport, maxExceptionsImportSize);
      if (this.enableServerExtensionPoints) {
        await this.serverExtensionsClient.pipeRun('exceptionsListPreImport', parsedObjects, this.getServerExtensionCallbackContext());
      }
      return (0, _import_exception_list_and_items.importExceptions)({
        exceptions: parsedObjects,
        generateNewListId: false,
        overwrite,
        savedObjectsClient,
        user
      });
    });
    /**
     * Opens a point in time (PIT) for either exception lists or exception list items.
     * See: https://www.elastic.co/guide/en/elasticsearch/reference/current/point-in-time-api.html
     * @param options
     * @param options.namespaceType "agnostic" or "single" depending on which namespace you are targeting
     * @param options.options The saved object PIT options
     * @returns The point in time (PIT)
     */
    (0, _defineProperty2.default)(this, "openPointInTime", async ({
      namespaceType,
      options
    }) => {
      const {
        savedObjectsClient
      } = this;
      return (0, _open_point_in_time.openPointInTime)({
        namespaceType,
        options,
        savedObjectsClient
      });
    });
    /**
     * Closes a point in time (PIT) for either exception lists or exception list items.
     * See: https://www.elastic.co/guide/en/elasticsearch/reference/current/point-in-time-api.html
     * @param options
     * @param options.pit The point in time to close
     * @returns The point in time (PIT)
     */
    (0, _defineProperty2.default)(this, "closePointInTime", async ({
      pit
    }) => {
      const {
        savedObjectsClient
      } = this;
      return (0, _close_point_in_time.closePointInTime)({
        pit,
        savedObjectsClient
      });
    });
    /**
     * Finds an exception list item within a point in time (PIT) and then calls the function
     * `executeFunctionOnStream` until the maxPerPage is reached and stops.
     * NOTE: This is slightly different from the saved objects version in that it takes
     * an injected function, so that we avoid doing additional plumbing with generators
     * to try to keep the maintenance of this machinery simpler for now.
     *
     * If you want to stream all results up to 10k into memory for correlation this would be:
     * @example
     * ```ts
     * const exceptionList: ExceptionListItemSchema[] = [];
     * const executeFunctionOnStream = (response: FoundExceptionListItemSchema) => {
     *   exceptionList = [...exceptionList, ...response.data];
     * }
     * await client.findExceptionListItemPointInTimeFinder({
     *   filter,
     *   executeFunctionOnStream,
     *   namespaceType,
     *   maxSize: 10_000, // NOTE: This is unbounded if it is "undefined"
     *   perPage: 1_000, // See https://github.com/elastic/kibana/issues/93770 for choice of 1k
     *   sortField,
     *   sortOrder,
     *   exe
     * });
     * ```
     * @param options
     * @param options.filter The filter to apply in the search
     * @param options.listId The "list_id" to filter against and find against
     * @param options.namespaceType "agnostic" | "single" of your namespace
     * @param options.perPage The number of items per page. Typical value should be 1_000 here. Never go above 10_000
     * @param options.maxSize If given a max size, this will not exceeded. Otherwise if undefined is passed down, all records will be processed.
     * @param options.sortField String of the field to sort against
     * @param options.sortOrder "asc" | "desc" The order to sort against, "undefined" if the order does not matter
     * @param options.executeFunctionOnStream The function to execute which will have the streamed results
     */
    (0, _defineProperty2.default)(this, "findExceptionListItemPointInTimeFinder", async ({
      executeFunctionOnStream,
      filter,
      listId,
      maxSize,
      namespaceType,
      perPage,
      sortField,
      sortOrder
    }) => {
      const {
        savedObjectsClient
      } = this;
      return (0, _find_exception_list_item_point_in_time_finder.findExceptionListItemPointInTimeFinder)({
        executeFunctionOnStream,
        filter,
        listId,
        maxSize,
        namespaceType,
        perPage,
        savedObjectsClient,
        sortField,
        sortOrder
      });
    });
    /**
     * Finds an exception list within a point in time (PIT) and then calls the function
     * `executeFunctionOnStream` until the maxPerPage is reached and stops.
     * NOTE: This is slightly different from the saved objects version in that it takes
     * an injected function, so that we avoid doing additional plumbing with generators
     * to try to keep the maintenance of this machinery simpler for now.
     *
     * If you want to stream all results up to 10k into memory for correlation this would be:
     * @example
     * ```ts
     * const exceptionList: ExceptionListSchema[] = [];
     * const executeFunctionOnStream = (response: FoundExceptionListSchema) => {
     *   exceptionList = [...exceptionList, ...response.data];
     * }
     * await client.findExceptionListPointInTimeFinder({
     *   filter,
     *   executeFunctionOnStream,
     *   namespaceType,
     *   maxSize: 10_000, // NOTE: This is unbounded if it is "undefined"
     *   perPage: 1_000, // See https://github.com/elastic/kibana/issues/93770 for choice of 1k
     *   sortField,
     *   sortOrder,
     *   exe
     * });
     * ```
     * @param options
     * @param options.filter The filter to apply in the search
     * @param options.namespaceType "agnostic" | "single" of your namespace
     * @param options.perPage The number of items per page. Typical value should be 1_000 here. Never go above 10_000
     * @param options.maxSize If given a max size, this will not be exceeded. Otherwise if undefined is passed down, all records will be processed.
     * @param options.sortField String of the field to sort against
     * @param options.sortOrder "asc" | "desc" The order to sort against, "undefined" if the order does not matter
     * @param options.executeFunctionOnStream The function to execute which will have the streamed results
     */
    (0, _defineProperty2.default)(this, "findExceptionListPointInTimeFinder", async ({
      executeFunctionOnStream,
      filter,
      maxSize,
      namespaceType,
      perPage,
      sortField,
      sortOrder
    }) => {
      const {
        savedObjectsClient
      } = this;
      return (0, _find_exception_list_point_in_time_finder.findExceptionListPointInTimeFinder)({
        executeFunctionOnStream,
        filter,
        maxSize,
        namespaceType,
        perPage,
        savedObjectsClient,
        sortField,
        sortOrder
      });
    });
    /**
     * Finds exception list items within a point in time (PIT) and then calls the function
     * `executeFunctionOnStream` until the maxPerPage is reached and stops.
     * NOTE: This is slightly different from the saved objects version in that it takes
     * an injected function, so that we avoid doing additional plumbing with generators
     * to try to keep the maintenance of this machinery simpler for now.
     *
     * If you want to stream all results up to 10k into memory for correlation this would be:
     * @example
     * ```ts
     * const exceptionList: ExceptionListItemSchema[] = [];
     * const executeFunctionOnStream = (response: FoundExceptionListItemSchema) => {
     *   exceptionList = [...exceptionList, ...response.data];
     * }
     * await client.findExceptionListsItemPointInTimeFinder({
     *   filter,
     *   executeFunctionOnStream,
     *   namespaceType,
     *   maxSize: 10_000, // NOTE: This is unbounded if it is "undefined"
     *   perPage: 1_000, // See https://github.com/elastic/kibana/issues/93770 for choice of 1k
     *   sortField,
     *   sortOrder,
     *   exe
     * });
     * ```
     * @param options
     * @param options.listId The "list_id" to find against
     * @param options.filter The filter to apply in the search
     * @param options.namespaceType "agnostic" | "single" of your namespace
     * @param options.perPage The number of items per page. Typical value should be 1_000 here. Never go above 10_000
     * @param options.maxSize If given a max size, this will not exceeded. Otherwise if undefined is passed down, all records will be processed.
     * @param options.sortField String of the field to sort against
     * @param options.sortOrder "asc" | "desc" The order to sort against, "undefined" if the order does not matter
     * @param options.executeFunctionOnStream The function to execute which will have the streamed results
     */
    (0, _defineProperty2.default)(this, "findExceptionListsItemPointInTimeFinder", async ({
      listId,
      namespaceType,
      executeFunctionOnStream,
      maxSize,
      filter,
      perPage,
      sortField,
      sortOrder
    }) => {
      const {
        savedObjectsClient
      } = this;
      return (0, _find_exception_list_items_point_in_time_finder.findExceptionListsItemPointInTimeFinder)({
        executeFunctionOnStream,
        filter,
        listId,
        maxSize,
        namespaceType,
        perPage,
        savedObjectsClient,
        sortField,
        sortOrder
      });
    });
    /**
     * Finds value lists within exception lists within a point in time (PIT) and then calls the function
     * `executeFunctionOnStream` until the maxPerPage is reached and stops.
     * NOTE: This is slightly different from the saved objects version in that it takes
     * an injected function, so that we avoid doing additional plumbing with generators
     * to try to keep the maintenance of this machinery simpler for now.
     *
     * If you want to stream all results up to 10k into memory for correlation this would be:
     * @example
     * ```ts
     * const exceptionList: ExceptionListItemSchema[] = [];
     * const executeFunctionOnStream = (response: FoundExceptionListItemSchema) => {
     *   exceptionList = [...exceptionList, ...response.data];
     * }
     * await client.findValueListExceptionListItemsPointInTimeFinder({
     *   valueListId,
     *   executeFunctionOnStream,
     *   namespaceType,
     *   maxSize: 10_000, // NOTE: This is unbounded if it is "undefined"
     *   perPage: 1_000, // See https://github.com/elastic/kibana/issues/93770 for choice of 1k
     *   sortField,
     *   sortOrder,
     *   exe
     * });
     * ```
     * @param options
     * @param options.valueListId The value list id
     * @param options.namespaceType "agnostic" | "single" of your namespace
     * @param options.perPage The number of items per page. Typical value should be 1_000 here. Never go above 10_000
     * @param options.maxSize If given a max size, this will not exceeded. Otherwise if undefined is passed down, all records will be processed.
     * @param options.sortField String of the field to sort against
     * @param options.sortOrder "asc" | "desc" The order to sort against, "undefined" if the order does not matter
     */
    (0, _defineProperty2.default)(this, "findValueListExceptionListItemsPointInTimeFinder", async ({
      valueListId,
      executeFunctionOnStream,
      perPage,
      maxSize,
      sortField,
      sortOrder
    }) => {
      const {
        savedObjectsClient
      } = this;
      return (0, _find_value_list_exception_list_items_point_in_time_finder.findValueListExceptionListItemsPointInTimeFinder)({
        executeFunctionOnStream,
        maxSize,
        perPage,
        savedObjectsClient,
        sortField,
        sortOrder,
        valueListId
      });
    });
    this.user = _user;
    this.savedObjectsClient = _savedObjectsClient;
    this.serverExtensionsClient = serverExtensionsClient;
    this.enableServerExtensionPoints = enableServerExtensionPoints;
    this.request = request;
  }
  getServerExtensionCallbackContext() {
    const {
      user,
      serverExtensionsClient,
      savedObjectsClient,
      request
    } = this;
    let exceptionListClient;
    return {
      // Lazy getter so that we only initialize a new instance of the class if needed
      get exceptionListClient() {
        if (!exceptionListClient) {
          exceptionListClient = new ExceptionListClient({
            enableServerExtensionPoints: false,
            request,
            savedObjectsClient,
            serverExtensionsClient,
            user
          });
        }
        return exceptionListClient;
      },
      request: this.request
    };
  }
}
exports.ExceptionListClient = ExceptionListClient;