"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerLensVisualizationsCreateAPIRoute = void 0;
var _boom = require("@hapi/boom");
var _constants = require("../../../../common/constants");
var _schema = require("./schema");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerLensVisualizationsCreateAPIRoute = (router, {
  contentManagement,
  builder
}) => {
  const createRoute = router.post({
    path: `${_constants.LENS_VIS_API_PATH}/{id?}`,
    access: _constants.LENS_API_ACCESS,
    enableQueryVersion: true,
    summary: 'Create Lens visualization',
    description: 'Create a new Lens visualization.',
    options: {
      tags: ['oas-tag:Lens'],
      availability: {
        stability: 'experimental'
      }
    },
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on Content Client for authorization'
      }
    }
  });
  createRoute.addVersion({
    version: _constants.LENS_API_VERSION,
    validate: {
      request: {
        query: _schema.lensCreateRequestQuerySchema,
        params: _schema.lensCreateRequestParamsSchema,
        body: _schema.lensCreateRequestBodySchema
      },
      response: {
        201: {
          body: () => _schema.lensCreateResponseBodySchema,
          description: 'Created'
        },
        400: {
          description: 'Malformed request'
        },
        401: {
          description: 'Unauthorized'
        },
        403: {
          description: 'Forbidden'
        },
        500: {
          description: 'Internal Server Error'
        }
      }
    }
  }, async (ctx, req, res) => {
    const requestBodyData = req.body;
    if ('state' in requestBodyData && !requestBodyData.visualizationType) {
      throw new Error('visualizationType is required');
    }

    // TODO fix IContentClient to type this client based on the actual
    const client = contentManagement.contentClient.getForRequest({
      request: req,
      requestHandlerContext: ctx
    }).for(_constants.LENS_CONTENT_TYPE);
    try {
      // Note: these types are to enforce loose param typings of client methods
      const {
        references,
        ...data
      } = (0, _utils.getLensRequestConfig)(builder, req.body);
      const options = {
        ...req.query,
        references,
        id: req.params.id
      };
      const {
        result
      } = await client.create(data, options);
      if (result.item.error) {
        throw result.item.error;
      }
      const responseItem = (0, _utils.getLensResponseItem)(builder, result.item);
      return res.created({
        body: responseItem
      });
    } catch (error) {
      if ((0, _boom.isBoom)(error) && error.output.statusCode === 403) {
        return res.forbidden();
      }
      return (0, _boom.boomify)(error); // forward unknown error
    }
  });
};
exports.registerLensVisualizationsCreateAPIRoute = registerLensVisualizationsCreateAPIRoute;