"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.subscribeToExternalContext = subscribeToExternalContext;
var _rxjs = require("rxjs");
var _lodash = require("lodash");
var _public = require("@kbn/data-plugin/public");
var _ = require("..");
var _utils = require("../../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * subscribes to external changes for filters, searchSessionId, timerange, autorefresh and projectRouting
 */
function subscribeToExternalContext({
  data,
  cps
}, getState, dispatch) {
  var _cps$cpsManager;
  const {
    query: queryService,
    search
  } = data;
  const {
    filterManager
  } = queryService;
  const dispatchFromExternal = (searchSessionId = search.session.start()) => {
    const globalFilters = filterManager.getFilters();
    const filters = (0, _lodash.isEqual)(getState().lens.filters, globalFilters) ? null : {
      filters: globalFilters
    };
    dispatch((0, _.setExecutionContext)({
      searchSessionId,
      ...filters,
      resolvedDateRange: (0, _utils.getResolvedDateRange)(queryService.timefilter.timefilter)
    }));
  };
  const debounceDispatchFromExternal = (0, _lodash.debounce)(dispatchFromExternal, 100);
  const sessionSubscription = search.session.getSession$()
  // wait for a tick to filter/timerange subscribers the chance to update the session id in the state
  .pipe((0, _rxjs.delay)(0))
  // then update if it didn't get updated yet
  .subscribe(newSessionId => {
    if (newSessionId && getState().lens.searchSessionId !== newSessionId) {
      debounceDispatchFromExternal(newSessionId);
    }
  });
  const filterSubscription = filterManager.getUpdates$().subscribe({
    next: () => {
      debounceDispatchFromExternal();
    }
  });
  const timeSubscription = data.query.timefilter.timefilter.getTimeUpdate$().subscribe({
    next: () => {
      debounceDispatchFromExternal();
    }
  });
  const autoRefreshSubscription = data.query.timefilter.timefilter.getAutoRefreshFetch$().pipe((0, _rxjs.tap)(() => {
    debounceDispatchFromExternal();
  }), (0, _rxjs.switchMap)(done =>
  // best way in lens to estimate that all panels are updated is to rely on search session service state
  (0, _public.waitUntilNextSessionCompletes$)(search.session).pipe((0, _rxjs.finalize)(done)))).subscribe();

  // Subscribe to CPS projectRouting changes
  const cpsProjectRoutingSubscription = cps === null || cps === void 0 ? void 0 : (_cps$cpsManager = cps.cpsManager) === null || _cps$cpsManager === void 0 ? void 0 : _cps$cpsManager.getProjectRouting$().subscribe(cpsProjectRouting => {
    const currentProjectRouting = getState().lens.projectRouting;
    if (cpsProjectRouting !== currentProjectRouting) {
      dispatch((0, _.setState)({
        projectRouting: cpsProjectRouting
      }));
    }
  });
  return () => {
    filterSubscription.unsubscribe();
    timeSubscription.unsubscribe();
    autoRefreshSubscription.unsubscribe();
    sessionSubscription.unsubscribe();
    cpsProjectRoutingSubscription === null || cpsProjectRoutingSubscription === void 0 ? void 0 : cpsProjectRoutingSubscription.unsubscribe();
  };
}