"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getBreakdownColumn = void 0;
var _date_histogram = require("./date_histogram");
var _top_values = require("./top_values");
var _intervals = require("./intervals");
var _filters = require("./filters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const DEFAULT_BREAKDOWN_SIZE = 5;
function getBreakdownType(field, dataview) {
  if (!dataview.fields.getByName(field)) {
    throw new Error(`field ${field} does not exist on dataview ${dataview.id ? dataview.id : dataview.title}`);
  }
  switch (dataview.fields.getByName(field).type) {
    case 'string':
      return 'topValues';
    case 'number':
      return 'intervals';
    case 'date':
      return 'dateHistogram';
    default:
      return 'topValues';
  }
}
const getBreakdownColumn = ({
  options,
  dataView
}) => {
  const breakdownType = typeof options === 'string' ? getBreakdownType(options, dataView) : options.type;
  const field = typeof options === 'string' ? options : 'field' in options ? options.field : '';
  const config = typeof options !== 'string' ? options : {};
  switch (breakdownType) {
    case 'dateHistogram':
      return (0, _date_histogram.getHistogramColumn)({
        options: {
          sourceField: field,
          params: typeof options !== 'string' ? {
            interval: options.minimumInterval || 'auto'
          } : {
            interval: 'auto'
          }
        }
      });
    case 'topValues':
      const topValuesOptions = config;
      return (0, _top_values.getTopValuesColumn)({
        field,
        options: {
          size: topValuesOptions.size || DEFAULT_BREAKDOWN_SIZE
        }
      });
    case 'intervals':
      const intervalOptions = config;
      return (0, _intervals.getIntervalsColumn)({
        field,
        options: {
          type: 'range',
          ranges: [{
            from: 0,
            to: 1000,
            label: ''
          }],
          maxBars: intervalOptions.granularity || 'auto'
        }
      });
    case 'filters':
      const filterOptions = config;
      return (0, _filters.getFiltersColumn)({
        options: {
          filters: filterOptions.filters.map(f => ({
            label: f.label || '',
            input: {
              language: 'kuery',
              query: f.filter
            }
          }))
        }
      });
  }
};
exports.getBreakdownColumn = getBreakdownColumn;